<?php

namespace Modules\BusinessDirectory\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Modules\BusinessDirectory\Entities\Bid;
use Illuminate\Support\Str;

class BidStatusChanged extends Notification implements ShouldQueue
{
    use Queueable;

    public Bid $bid;
    public string $oldStatus;
    public string $newStatus;

    /**
     * Create a new notification instance.
     *
     * @param Bid $bid
     * @param string $oldStatus
     * @param string $newStatus
     */
    public function __construct(Bid $bid, string $oldStatus, string $newStatus)
    {
        $this->bid = $bid;
        $this->oldStatus = $oldStatus;
        $this->newStatus = $newStatus;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
            return ['database', 'mail'];
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        $tender = $this->bid->tender;
        $message = "The status of your bid for the tender \"{$tender->title}\" has been updated from " .
                   Str::title(str_replace('_', ' ', $this->oldStatus)) . " to " .
                   Str::title(str_replace('_', ' ', $this->newStatus)) . ".";

        if ($this->newStatus === 'accepted') {
            $message = "Congratulations! Your bid for the tender \"{$tender->title}\" has been accepted!";
        } elseif ($this->newStatus === 'rejected') {
            $message = "Regarding your bid for the tender \"{$tender->title}\", it was not selected this time. Status: Rejected.";
        }

        return [
            'tender_id' => $tender->id,
            'tender_title' => $tender->title,
            'tender_slug' => $tender->slug,
            'bid_id' => $this->bid->id,
            'old_status' => $this->oldStatus,
            'new_status' => $this->newStatus,
            'message' => $message,
            'link' => route('public.businessdirectory.tenders.show', $tender->slug) . '#bid-' . $this->bid->id,
        ];
    }

    // Optional: toMail method
        public function toMail($notifiable)
        {
            $tender = $this->bid->tender;
            $url = route('public.businessdirectory.tenders.show', $tender->slug) . '#bid-' . $this->bid->id;
            $subject = 'Your Bid Status Updated: ' . Str::limit($tender->title, 40);
            $greeting = 'Hello ' . $notifiable->name . ',';
            $line = $this->toArray($notifiable)['message']; // Reuse the generated message

            return (new MailMessage)
                        ->subject($subject)
                        ->greeting($greeting)
                        ->line($line)
                        ->action('View Bid Details', $url)
                        ->line('Thank you for using our platform.');
        }
}
