<?php

namespace Modules\BusinessDirectory\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Tender;

class StoreTenderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        // Authorization to create a tender for a specific company is handled in the controller
        // using $this->authorize('createForCompany', [Tender::class, $selectedCompany]);
        // This basic check ensures the user is logged in.
        // More granular checks (like owning the selected company) are in the controller.
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $user = Auth::user();
        $userCompanyIds = $user->companies()->where('status', 'approved')->pluck('id')->toArray();

        return [
            'title' => 'required|string|max:255',
            'company_id' => ['required', Rule::in($userCompanyIds)],
            'description' => 'required|string|max:10000',
            'requirements' => 'nullable|string|max:5000',
            'budget_min' => 'nullable|numeric|min:0',
            'budget_max' => 'nullable|numeric|min:0|gte:budget_min',
            'currency' => 'nullable|string|max:3',
            'deadline' => 'required|date|after_or_equal:today',
            'type' => ['nullable', 'string', Rule::in(array_keys(Tender::getTenderTypes()))],
            'location' => 'nullable|string|max:255',
            'attachments' => 'nullable|array',
            'attachments.*' => [
                'file',
                'mimes:pdf,doc,docx,xls,xlsx,ppt,pptx,txt,jpg,jpeg,png,zip',
                'max:5120' // Max 5MB per file
            ],
        ];
    }
}