<?php

namespace Modules\BusinessDirectory\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyProject;

class StoreCompanyProjectRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        // The controller's store method will perform a more specific authorization check:
        // $this->authorize('create', [CompanyProject::class, $company]);
        // This basic check ensures the user is logged in and can generally create projects
        // if they have an eligible company.
        $company = Company::find($this->input('company_id'));
        if (!$company) {
            return false; // No company selected or found
        }
        // Check if user owns the company they are trying to create a project for
        return $company->user_id === Auth::id();
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $user = Auth::user();
        $userCompanyIds = $user->companies()->where('status', 'approved')->pluck('id')->toArray();

        return [
            'company_id' => ['required', Rule::in($userCompanyIds)],
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:10000',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => ['required', 'string', Rule::in(array_keys(CompanyProject::getStatuses()))],
            'team_id' => ['nullable', 'integer', Rule::exists('teams', 'id')->where(function ($query) {
                $query->where('user_id', Auth::id()); // Ensure the team is owned by the current user
            })],
            // Add other fields like budget, currency, client_company_id if needed
        ];
    }
}