<?php

namespace Modules\BusinessDirectory\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Tender;

class StoreTenderRequest extends FormRequest
{
    public function authorize(): bool
    {
        return Auth::check() && Auth::user()->isAdmin();
    }

    public function rules(): array
    {
        return [
            'company_id' => 'required|exists:bd_companies,id',
            'user_id' => 'nullable|exists:users,id', // Admin can assign, else defaults to company owner
            'title' => ['required', 'string', 'max:255', Rule::unique('bd_tenders', 'title')],
            'description' => 'required|string|max:20000',
            'requirements' => 'nullable|string|max:10000',
            'budget_min' => 'nullable|numeric|min:0|lte:budget_max',
            'budget_max' => 'nullable|numeric|min:0|gte:budget_min',
            'budget_currency' => 'nullable|string|max:10',
            'deadline' => 'required|date|after_or_equal:today',
            'status' => ['required', 'string', Rule::in(array_keys(Tender::getStatuses()))],
            'location' => 'nullable|string|max:255',
            'category_id' => 'nullable|exists:bd_tender_categories,id', // Assuming you have TenderCategory model/table
            'tags' => 'nullable|string|max:255',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|mimes:pdf,doc,docx,xls,xlsx,jpg,png|max:5120', // Max 5MB per file
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:1000',
            'meta_keywords' => 'nullable|string|max:255',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'user_id' => $this->user_id ?? $this->company_id ? \Modules\BusinessDirectory\Entities\Company::find($this->company_id)?->user_id : null,
        ]);
    }
}