<?php

namespace Modules\BusinessDirectory\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Modules\BusinessDirectory\Entities\Product;

class StoreProductRequest extends FormRequest
{
    public function authorize(): bool
    {
        return Auth::check() && Auth::user()->isAdmin();
    }

    public function rules(): array
    {
        $companyId = $this->route('company')->id; // Get company from route

        return [
            'name' => ['required', 'string', 'max:255'],
            'description' => 'nullable|string|max:20000',
            'price' => 'required|numeric|min:0',
            'sku' => ['nullable', 'string', 'max:100', Rule::unique('bd_products', 'sku')->where('company_id', $companyId)],
            'stock_quantity' => 'nullable|integer|min:0',
            'status' => ['required', 'string', Rule::in(array_keys(Product::getStatuses()))],
            'is_published' => 'nullable|boolean',
            'featured_image' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'gallery_images' => 'nullable|array',
            'gallery_images.*' => ['image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            // 'category_id' => 'nullable|exists:bd_product_categories,id', // If using categories
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:1000',
            'meta_keywords' => 'nullable|string|max:255',
            'user_id' => 'nullable|exists:users,id', // Admin can assign, otherwise defaults to company owner
            'tags' => 'nullable|string|max:255', // Example: comma-separated tags
            'weight' => 'nullable|numeric|min:0',
            'dimensions.length' => 'nullable|numeric|min:0',
            'dimensions.width' => 'nullable|numeric|min:0',
            'dimensions.height' => 'nullable|numeric|min:0',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'is_published' => $this->boolean('is_published'),
            'dimensions' => [ // Ensure dimensions is an array even if sub-fields are empty
                'length' => $this->input('dimensions.length'),
                'width' => $this->input('dimensions.width'),
                'height' => $this->input('dimensions.height'),
            ],
        ]);
    }
}