<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\BusinessDirectory\Entities\Tender;
use Illuminate\Support\Facades\Auth;
use Modules\BusinessDirectory\Entities\Company;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str; // Needed for Str::title
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // For $this->authorize()
use App\Helpers\UserAccessHelper; // Import the helper
use Illuminate\Support\Facades\Storage; // For file storage
use Modules\BusinessDirectory\Entities\Attachment; // For creating attachment records
use Modules\BusinessDirectory\Entities\Bid; // Needed for updateBidStatus
use Modules\BusinessDirectory\Entities\Category; 

class TenderController extends Controller
{
    use AuthorizesRequests; // Use the trait for $this->authorize()

    // Public Tender Listings
    public function index(Request $request) // Add Request $request
    {
        $query = Tender::with(['company', 'category']) // Eager load category
                       ->where('status', Tender::STATUS_OPEN) // Use constant
                       ->where('deadline', '>=', now()) // Ensure only active tenders
                       ->latest();

        if ($request->filled('keywords')) {
            $keywords = $request->input('keywords');
            $query->where(function ($q) use ($keywords) {
                $q->where('title', 'like', "%{$keywords}%")
                  ->orWhere('description', 'like', "%{$keywords}%")
                  ->orWhereHas('company', function ($cq) use ($keywords) {
                      $cq->where('name', 'like', "%{$keywords}%");
                  });
            });
        }

        if ($request->filled('location')) {
            $query->where('location', 'like', '%' . $request->input('location') . '%');
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->input('category_id'));
        }

        $tenders = $query->paginate(setting('pagination_count', 10))->withQueryString();
        $categories = Category::orderBy('name')->get(); // Fetch categories for the filter

        $pageTitle = __('Tender Listings');
        $metaDescription = __('Explore current tender opportunities and submit your bids.');

        return view('businessdirectory::frontend.tenders.index', compact('tenders', 'categories', 'pageTitle', 'metaDescription'));
    }

    /**
     * Display the specified public tender.
     *
     * @param Tender $tender
     * @return \Illuminate\Contracts\View\View
     */
    public function show(Tender $tender) // Route model binding by slug (defined in routes/web.php)
    {
        // Authorization is handled by TenderPolicy@view, which should consider tender status and user roles.
        $this->authorize('view', $tender);

        $tender->load('company', 'bids.user', 'additionalAttachments'); // Load company, bids with bidder info, and additional attachments

        $pageTitle = $tender->title;
        $metaDescription = Str::limit(strip_tags($tender->description), 160);

        return view('businessdirectory::frontend.tenders.show', compact('tender', 'pageTitle', 'metaDescription'));
    }

    public function create(Request $request)
    {   
        $user = Auth::user();
        // Fetch companies owned by the user that are approved
        $userCompanies = $user->companies()
                               ->where('status', 'approved') // Only allow posting for approved companies
                               ->orderBy('name')
                               ->pluck('name', 'id');

        if ($userCompanies->isEmpty()) {
            return redirect()->route('frontend.businessdirectory.my-companies.index') // Or a more general page
                             ->with('warning', 'You need to have an approved company listing to post a tender. Please add or claim your company first.');
        }
        // Authorize creating a tender
        $this->authorize('create', Tender::class); // General ability to create

        if (!UserAccessHelper::checkLimit(Auth::user(), 'businessdirectory_post_tenders', 0, false)) {
            return redirect()->route('frontend.businessdirectory.my-tenders.index')
                             ->with('error', 'You have reached the maximum limit for posting tenders under your current plan.');
        }

        $tender = new Tender(); // For form model binding
        return view('businessdirectory::frontend.tenders.create', compact('tender', 'userCompanies'));
    }

    /**
     * Store a newly created tender in storage.
     * This method handles the 'frontend.businessdirectory.my-tenders.store' route.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(\Modules\BusinessDirectory\Http\Requests\StoreTenderRequest $request) // Use FormRequest
    {
        $user = Auth::user();
        // Authorize creating a tender for the selected company
        $selectedCompany = Company::findOrFail($request->input('company_id'));
        $this->authorize('createForCompany', [Tender::class, $selectedCompany]);

        if (!UserAccessHelper::checkLimitAndDecrement(Auth::user(), 'businessdirectory_post_tenders')) {
            return redirect()->route('frontend.businessdirectory.my-tenders.create')
                             ->with('error', 'You have reached the maximum limit for posting tenders. Please upgrade your subscription.')
                             ->withInput();
        }


        $validatedData = $request->validated();

        $validatedData['user_id'] = $user->id; // Set the user who created it
        $validatedData['status'] = 'open'; // Default status for new tenders

        // Remove attachments from validatedData as it's handled separately
        $attachmentFiles = $request->file('attachments') ?? []; // Get files directly from request
        unset($validatedData['attachments']);

        $tender = Tender::create($validatedData);

        // Handle attachments
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $filePath = $file->store("tenders/{$tender->id}/attachments", 'public');
                $tender->additionalAttachments()->create([ // Use the new relationship name
                    'user_id' => Auth::id(),
                    'file_path' => $filePath,
                    'original_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'size' => $file->getSize(),
                ]);
            }
        }

        // Notify relevant users (e.g., admin, followers of the company) about the new tender
        event(new \Modules\BusinessDirectory\Events\NewTenderPosted($tender));
        // Redirect to the 'My Tenders' index page
        return redirect()->route('frontend.businessdirectory.my-tenders.index')
                         ->with('success', 'Your tender has been posted successfully!');
    }

    public function edit(Tender $tender)
    {
        $this->authorize('update', $tender);
        $user = Auth::user();
        $userCompanies = $user->companies()
                               ->where('status', 'approved')
                               ->orderBy('name')
                               ->pluck('name', 'id');
        
        $tender->load('additionalAttachments'); // Corrected relationship name

        return view('businessdirectory::frontend.tenders.edit', compact('tender', 'userCompanies'));
    }

    public function update(\Modules\BusinessDirectory\Http\Requests\UpdateTenderRequest $request, Tender $tender)
    {
        $this->authorize('update', $tender);
        $validatedData = $request->validated();

        // Remove attachments from validatedData as it's handled separately
        $attachmentFiles = $request->file('attachments') ?? []; // Get files directly from request
        unset($validatedData['attachments']);

        $tender->update($validatedData);

        // Handle new attachments
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $filePath = $file->store("tenders/{$tender->id}/attachments", 'public');
                $tender->additionalAttachments()->create([ // Use the new relationship name
                    'user_id' => Auth::id(),
                    'file_path' => $filePath,
                    'original_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'size' => $file->getSize(),
                ]);
            }
        }
        // TODO: Implement deletion of existing attachments if needed

        return redirect()->route('frontend.businessdirectory.my-tenders.index')
                         ->with('success', 'Tender updated successfully.');
    }

    /**
     * Remove the specified tender from storage.
     * This method handles the 'frontend.businessdirectory.my-tenders.destroy' route.
     *
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Tender $tender)
    {
        $this->authorize('delete', $tender);

        // Delete associated attachments from storage and database
        foreach ($tender->additionalAttachments as $attachment) {
            Storage::disk('public')->delete($attachment->file_path);
            $attachment->delete();
        }
        // Consider implications for bids on this tender (e.g., notify bidders, archive bids)
        // For now, we'll assume bids might be soft-deleted or handled by DB constraints if tender is hard-deleted.

        $tender->delete(); // Soft delete if SoftDeletes trait is used on Tender model

        return redirect()->route('frontend.businessdirectory.my-tenders.index')
                         ->with('success', 'Tender deleted successfully.');
    }

    /**
     * Display a listing of tenders posted by the authenticated user's companies.
     * This method handles the 'frontend.businessdirectory.my-tenders.index' route.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function myTenders(Request $request)
    {
        $user = Auth::user();
        // Fetch IDs of companies owned by the user that are approved
        $userCompanyIds = $user->companies()->where('status', 'approved')->pluck('id')->toArray();

        // If the user has no approved companies, they can't have posted tenders
        if (empty($userCompanyIds)) {
             $myTenders = collect(); // Return an empty collection
        } else {
            $query = Tender::whereIn('company_id', $userCompanyIds)
                           ->withCount('bids') // Count bids directly
                           ->orderBy('created_at', 'desc');

            if ($request->filled('search')) {
                $searchTerm = $request->input('search');
                $query->where('title', 'like', "%{$searchTerm}%");
            }
    
            if ($request->filled('status')) {
                $query->where('status', $request->input('status'));
            }
            
            $myTenders = $query->paginate(10);
        }

        return view('businessdirectory::frontend.tenders.my_tenders', compact('myTenders'));
    }

    /**
     * Display a specific tender posted by the authenticated user.
     * This method handles the 'frontend.businessdirectory.my-tenders.show' route.
     *
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showMyTender(Tender $tender)
    {
        $this->authorize('view', $tender); // Ensure the user can view this tender (owns it)
        $tender->load('company', 'bids.user', 'additionalAttachments');
        return view('businessdirectory::frontend.tenders.show_my_tender', compact('tender')); // You'll need to create this view
    }
    /**
     * Update the status of a specific bid related to a tender owned by the user.
     * This method handles the 'frontend.businessdirectory.my-bids.update-status' route.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Modules\BusinessDirectory\Entities\Bid  $bid
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function updateBidStatus(Request $request, Bid $bid)
    {
        // Authorization is now primarily handled by TenderPolicy's manageBids (or similar)
        $this->authorize('manageBids', $bid->tender);

        $validatedStatus = $request->validate([
            'status' => ['required', Rule::in(array_keys(Bid::getStatuses()))], // Validate against defined Bid statuses
        ]);

        $bid->update(['status' => $validatedStatus['status']]);

        // Optional: If bid status is 'accepted', update the tender status to 'awarded'
        if ($validatedStatus['status'] === 'accepted') {
            $tender = $bid->tender;
            $tender->update(['status' => 'awarded', 'awarded_bid_id' => $bid->id]);
            // Notify other bidders that the tender has been awarded
            event(new \Modules\BusinessDirectory\Events\TenderAwardedToAnotherBidder($tender, $bid));
        }
        // Notify bidder about their bid status change
        event(new \Modules\BusinessDirectory\Events\BidStatusUpdated($bid, $validatedStatus['status']));

        // Redirect back to the tender show page
        return redirect()->route('frontend.businessdirectory.tenders.show', $bid->tender->slug)
                         ->with('success', "Bid status updated to '" . Str::title(str_replace('_', ' ', $validatedStatus['status'])) . "'.");
    }
}