<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\BusinessDirectory\Entities\Job;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Http\Requests\StoreJobRequest; // Create this
use Modules\BusinessDirectory\Http\Requests\UpdateJobRequest; // Create this
use Modules\BusinessDirectory\Entities\JobApplication;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // For $this->authorize()
use App\Helpers\UserAccessHelper; // Import the helper
use Illuminate\Support\Facades\Storage; // For file storage
use Modules\BusinessDirectory\Entities\Attachment; // For creating attachment records

class JobController extends Controller
{
    use AuthorizesRequests;

    /**
     * Display a listing of open jobs.
     */
    public function index(Request $request)
    {
        $query = Job::with('company')->where('status', 'open')->latest();

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhereHas('company', function ($companyQuery) use ($request) {
                      $companyQuery->where('name', 'like', '%' . $request->search . '%');
                  });
            });
        }

        if ($request->filled('location')) {
            $query->where('location', 'like', '%' . $request->location . '%');
        }

        if ($request->filled('job_type')) {
            $query->where('job_type', $request->job_type);
        }

        $jobs = $query->paginate(10);
        $job_types = Job::getJobTypes(); // Use the static method for consistency

        return view('businessdirectory::frontend.jobs.index', compact('jobs', 'job_types'));
    }

    /**
     * Display the specified job.
     * @param string $slug
     */
    public function show(string $slug)
    {
        $job = Job::with(['company', 'attachments']) // Eager load attachments
            ->where('slug', $slug)
            ->where('status', Job::STATUS_OPEN)->firstOrFail(); // Use constant
        $job->increment('views_count');

        $hasApplied = false;
        if (Auth::check()) {
            $hasApplied = JobApplication::where('job_id', $job->id)
                                        ->where('user_id', Auth::id())
                                        ->exists();
        }
        $relatedJobs = Job::with('company')
            ->where('status', 'open')
            ->where('company_id', $job->company_id)
            ->where('id', '!=', $job->id)
            ->latest()
            ->take(3)
            ->get();

        return view('businessdirectory::frontend.jobs.show', compact('job', 'hasApplied', 'relatedJobs'));
    }

    /**
     * Show jobs posted by the authenticated user's company.
     */
    public function myJobs(Request $request)
    {
        // Multi-Company Management: If a user can manage jobs for multiple companies,
        // this logic will need to be enhanced to allow company selection.
        $company = Auth::user()->companies()->first(); // Assumes User model has a 'companies' relationship
        if (!$company) {
            // Redirect to company creation/selection if no company is found
            return redirect()->route('frontend.businessdirectory.my-companies.index')->with('warning', 'You need to have a company profile to manage jobs. Please create or select a company.');
        }

        $query = Job::where('company_id', $company->id)->latest();

        // Apply Search Filter
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where('title', 'like', "%{$searchTerm}%");
        }

        // Apply Status Filter
        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        $jobs = $query->paginate(10);
        return view('businessdirectory::frontend.jobs.my_jobs', compact('jobs', 'company'));
    }

    /**
     * Show the form for creating a new job for the user's company.
     */
    public function createMyJob()
    {
        $company = Auth::user()->companies()->first();
        if (!$company) {
            return redirect()->route('frontend.businessdirectory.my-companies.index')->with('warning', 'You must have a company profile to post a job.');
        }
        $this->authorize('create', [Job::class, $company]); // Policy: Can user create job for this company?

        if (!UserAccessHelper::checkLimit(Auth::user(), 'businessdirectory_post_jobs', 0, false)) {
            return redirect()->route('frontend.businessdirectory.my_jobs')
                             ->with('error', 'You have reached the maximum limit for posting jobs under your current plan.');
        }
        $job_statuses = ['open' => 'Open', 'draft' => 'Draft']; // Company can only set these initially
        $job_types = ['full-time' => 'Full-time', 'part-time' => 'Part-time', 'contract' => 'Contract', 'internship' => 'Internship', 'temporary' => 'Temporary'];
        return view('businessdirectory::frontend.jobs.create_my_job', compact('company', 'job_statuses', 'job_types'));
    }

    /**
     * Store a newly created job for the user's company.
     */
    public function storeMyJob(StoreJobRequest $request) // Use Form Request
    {
        $company = Auth::user()->companies()->first();
        if (!$company) {
            return redirect()->back()->with('error', 'You must have a company profile to post a job.')->withInput();
        }
        $this->authorize('create', [Job::class, $company]);

        if (!UserAccessHelper::checkLimitAndDecrement(Auth::user(), 'businessdirectory_post_jobs')) {
            return redirect()->route('frontend.businessdirectory.my_jobs.create')
                             ->with('error', 'You have reached the maximum limit for posting jobs. Please upgrade your subscription.')
                             ->withInput();
        }

        // Validation is handled by StoreJobRequest
        $validatedData = $request->validated();

        // Remove attachments from validatedData as it's handled separately
        $attachmentFiles = $validatedData['attachments'] ?? [];
        unset($validatedData['attachments']);
        
        $job = new Job($validatedData);
        $job->company_id = $company->id;
        $job->user_id = Auth::id(); // Set the user_id for the job poster
        // $job->status = $validatedData['status'] ?? 'draft'; // Or set a default like 'pending_approval' if admin must approve

        $job->save();

        // Handle attachments
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $filePath = $file->store("jobs/{$job->id}/attachments", 'public');
                $job->attachments()->create([
                    'user_id' => Auth::id(),
                    'file_path' => $filePath,
                    'original_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'size' => $file->getSize(),
                ]);
            }
        }

        return redirect()->route('frontend.businessdirectory.my_jobs')->with('success', 'Job posted successfully.');
    }

     /**
     * Show the form for editing a job for the user's company.
     */
    public function editMyJob(Job $job)
    {
        $this->authorize('manage', $job); // Policy check
        $company = $job->company; // Define $company
        $job_statuses = ['open' => 'Open', 'closed' => 'Closed', 'filled' => 'Filled', 'draft' => 'Draft'];
        $job->load('attachments'); // Eager load existing attachments
        $job_types = ['full-time' => 'Full-time', 'part-time' => 'Part-time', 'contract' => 'Contract', 'internship' => 'Internship', 'temporary' => 'Temporary'];
        return view('businessdirectory::frontend.jobs.edit_my_job', compact('job', 'company', 'job_statuses', 'job_types'));
    }

    /**
     * Update a job for the user's company.
     */
    public function updateMyJob(UpdateJobRequest $request, Job $job) // Use Form Request
    {
        $this->authorize('manage', $job); // Policy check
        $validatedData = $request->validated();

        // Remove attachments from validatedData as it's handled separately
        $attachmentFiles = $validatedData['attachments'] ?? [];
        unset($validatedData['attachments']);

        $job->update($validatedData);

        // Handle new attachments
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $filePath = $file->store("jobs/{$job->id}/attachments", 'public');
                $job->attachments()->create([
                    'user_id' => Auth::id(),
                    'file_path' => $filePath,
                    'original_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'size' => $file->getSize(),
                ]);
            }
        }
        // TODO: Implement deletion of existing attachments if needed

        return redirect()->route('frontend.businessdirectory.my_jobs')->with('success', 'Job updated successfully.');
    }

    /**
     * Remove the specified job from storage.
     */
    public function destroyMyJob(Job $job)
    {
        $this->authorize('delete', $job); // Policy check (assuming 'delete' method in JobPolicy)
        // Deletion Checks: Consider if a job with active applications can be deleted,
        // or if applications should be archived/notified.
        // if ($job->applications()->exists()) {
        // }
        
        // Delete associated attachments from storage and database
        foreach ($job->attachments as $attachment) {
            Storage::disk('public')->delete($attachment->file_path);
            $attachment->delete();
        }

        $job->delete(); // This will soft delete if SoftDeletes trait is used on Job model

        return redirect()->route('frontend.businessdirectory.my_jobs')->with('success', 'Job deleted successfully.');
    }
}