<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use App\Http\Controllers\Controller; // Assuming your base controller is in App\Http\Controllers
use Illuminate\Http\Request;
use Modules\BusinessDirectory\Entities\CompanyType;
use Modules\BusinessDirectory\Helpers\FeaturedContentHelper;
use Modules\BusinessDirectory\Entities\Category;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Job;
use Modules\BusinessDirectory\Entities\Product;

class HomepageController extends Controller
{
    public function index(Request $request)
    {
        // Page Meta
        $pageTitle = setting('homepage_meta_title', setting('site_name', 'Welcome'));
        $metaDescription = setting('homepage_meta_description', 'Find the best businesses, products, and services.');
        $metaKeywords = setting('homepage_meta_keywords', 'business, directory, products, services, local');

        // Fetch content for main sections using the helper
        $featuredCompaniesLimit = (int) setting('homepage_featured_companies_count', 6);
        $featuredProductsLimit = (int) setting('homepage_featured_products_count', 8);
        $latestTendersLimit = (int) setting('homepage_latest_tenders_count', 5);
        $latestJobsLimit = (int) setting('homepage_latest_jobs_count', 5);

        // Fetch featured companies and eager load reviews to calculate average rating
        $featuredCompanies = Company::approved()->featured()
            ->withAvg(['reviews' => function ($query) {
                $query->where('is_approved', true);
            }], 'rating')
            ->withCount(['reviews' => function ($query) {
                $query->where('is_approved', true);
            }])
            ->take($featuredCompaniesLimit)->get();

        // If no featured companies, get some random approved ones
        if ($featuredCompanies->isEmpty() && $featuredCompaniesLimit > 0) {
            $featuredCompanies = Company::approved()
                ->withAvg(['reviews' => function ($query) { $query->where('is_approved', true); }], 'rating')
                ->withCount(['reviews' => function ($query) { $query->where('is_approved', true); }])
                ->inRandomOrder()
                ->take($featuredCompaniesLimit)
                ->get();
        }
        $featuredProducts = FeaturedContentHelper::getFeaturedProducts($featuredProductsLimit);
        $latestTenders = FeaturedContentHelper::getLatestTenders($latestTendersLimit);
        $latestJobs = FeaturedContentHelper::getLatestJobs($latestJobsLimit);

        $companyTypes = CompanyType::orderBy('name')->get();

        // --- Data for Sidebar ---
        $sidebarCategories = Category::query()
            // ->withCount([ /* Add counts if needed and performance allows */ ])
            ->whereNull('parent_id') // Root categories
            ->orderBy('name')
            ->get();

        $sidebarFeaturedCompanies = Company::approved()->featured()
                                        ->with('companyType')
                                        // ->withAvg('reviews', 'rating') // Optionally add ratings here too
                                        ->inRandomOrder()
                                        ->take(3)
                                        ->get();

        $sidebarLatestTenders = Tender::open() // Assuming Tender::scopeOpen()
                                    ->latest()
                                    ->take(5)
                                    ->get();

        // Fetch recent activities (simplified example)
        $recentTendersCollection = Tender::open()->latest()->take(2)->get();
        $recentJobsCollection = Job::open()->latest()->take(2)->get(); // Assuming Job::scopeOpen()
        $recentProductsCollection = Product::published()->latest()->take(2)->get(); // Assuming Product::scopePublished()
        $newCompaniesCollection = Company::approved()->latest()->take(1)->get();

        $recentActivities = collect()
            ->merge($recentTendersCollection)
            ->merge($recentJobsCollection)
            ->merge($recentProductsCollection)
            ->merge($newCompaniesCollection)
            ->sortByDesc('created_at')
            ->take(5);

        return view('businessdirectory::frontend.home', compact(
            'pageTitle', 'metaDescription', 'metaKeywords',
            'featuredCompanies', 'featuredProducts', 'latestTenders', 'latestJobs', 'companyTypes',
            'sidebarCategories',
            'sidebarFeaturedCompanies',
            'sidebarLatestTenders',
            'recentActivities'
        ));
    }

    public function subscribeNewsletter(Request $request)
    {
        $request->validate(['email' => 'required|email|unique:newsletter_subscriptions,email']); // Assuming a newsletter_subscriptions table

        // Logic to store the email
        // NewsletterSubscription::create(['email' => $request->email]); // Example

        return back()->with('success', 'Thank you for subscribing to our newsletter!');
    }

    public function showOpportunityCreatePage()
    {
        // This page could list options like "Create Company", "Post Job", "Post Tender"
        // For now, a simple placeholder view.
        return view('businessdirectory::frontend.opportunity_create_placeholder');
    }

    // Placeholder for all categories page
    public function allCategories()
    {
        $categories = Category::orderBy('name')->paginate(20); // Example
        // Potentially load $pageTitle, $metaDescription as well
        return view('businessdirectory::frontend.categories.index', compact('categories'));
    }

    // Placeholder for showing items in a specific category
    public function showCategory(Category $category)
    {
        // Fetch companies, products, etc., related to this category
        $companies = $category->companies()->approved()->paginate(10); // Example
        // Potentially load $pageTitle, $metaDescription as well
        return view('businessdirectory::frontend.categories.show', compact('category', 'companies'));
    }

}