<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // For $this->authorize()
use Modules\BusinessDirectory\Entities\Product;
use App\Helpers\UserAccessHelper; // Import the helper
use Modules\BusinessDirectory\Entities\Company;
// Import necessary Form Requests if you create them
use Modules\BusinessDirectory\Http\Requests\StoreProductRequest;
use Modules\BusinessDirectory\Http\Requests\UpdateProductRequest;

class FrontendProductController extends Controller
{
    use AuthorizesRequests; // Use the trait

    public function __construct()
    {
        // Apply auth middleware to management methods
        $this->middleware('auth')->except(['index', 'show']);
    }

    /**
     * Display a listing of published products.
     */
    public function index(Request $request)
    {
        // TODO: Implement comprehensive public product listing with advanced search/filters (categories, price range, etc.)
        $products = Product::where('is_published', true)->paginate(12);
        return view('businessdirectory::frontend.products.index', compact('products'));
    }

    /**
     * Display the specified product.
     */
    public function show(Product $product)
    {
        $this->authorize('view', $product);
        if (!$product->is_published) {
             // Optionally allow owner/admin to view unpublished products
             abort(404); // Or redirect
        }
        $product->load('company'); // Eager load company
        return view('businessdirectory::frontend.products.show', compact('product'));
    }

    public function myProductsIndex(Request $request)
    {
        $user = auth()->user();
        // Fetch products belonging to companies owned by the authenticated user
        $userCompanyIds = $user->companies()->pluck('id')->toArray();

        $query = Product::whereIn('company_id', $userCompanyIds)
                        ->with('company'); // Eager load company

        // Add filtering/searching
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%");
        }

        if ($request->filled('status')) {
             // Assuming 'is_published' or a separate status field
             $query->where('is_published', $request->status === 'published'); // Example
        }

        if ($request->filled('company_id')) {
             // Ensure the requested company_id is one the user owns
             if (in_array($request->company_id, $userCompanyIds)) {
                 $query->where('company_id', $request->company_id);
             }
        }

        $products = $query->latest()->paginate(10);

        // You might also need to pass the user's companies for a filter dropdown
        $userCompanies = $user->companies()->orderBy('name')->pluck('name', 'id');

        return view('businessdirectory::frontend.products.my_index', compact('products', 'userCompanies'));
    }

    /**
     * Show the form for creating a new product for one of the user's companies.
     */
    public function create()
    {
        $user = Auth::user();
        $companies = $user->companies()->where('status', 'approved')->orderBy('name')->pluck('name', 'id');

        if ($companies->isEmpty()) {
            return redirect()->route('frontend.businessdirectory.my-companies.index')
                             ->with('warning', 'You need an approved company to post a product.');
        }
        // Authorize based on the ability to create products in general, specific company check in store
        $this->authorize('create', Product::class);

        if (!UserAccessHelper::checkLimit(Auth::user(), 'businessdirectory_manage_products', 0, false)) {
            return redirect()->route('frontend.businessdirectory.my-products.index')
                             ->with('error', 'You have reached the maximum limit for creating products under your current plan.');
        }

        // TODO: Add other necessary data like categories, product types, etc.
        return view('businessdirectory::frontend.products.create', compact('companies'));
    }

    /**
     * Store a newly created product in storage.
     */
    public function store(StoreProductRequest $request) // Assuming StoreProductRequest exists
    {
        $user = Auth::user();
        $validatedData = $request->validated();

        $company = $user->companies()->findOrFail($validatedData['company_id']);
        $this->authorize('create', [Product::class, $company]); // Authorize creating product for this specific company

        if (!UserAccessHelper::checkLimitAndDecrement(Auth::user(), 'businessdirectory_manage_products')) {
            return redirect()->route('frontend.businessdirectory.my-products.create')
                             ->with('error', 'You have reached the maximum limit for creating products. Please upgrade your subscription.')
                             ->withInput();
        }

        // Handle image uploads if any
        // if ($request->hasFile('featured_image')) {
        //     $validatedData['featured_image'] = $request->file('featured_image')->store('products', 'public');
        // }

        $product = $company->products()->create($validatedData + ['user_id' => $user->id]);

        return redirect()->route('frontend.businessdirectory.my-products.index')
                         ->with('success', 'Product created successfully.');
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Product $product)
    {
        $this->authorize('update', $product);
        $user = Auth::user();
        // Ensure the product belongs to one of the user's companies
        if (!$user->companies()->where('id', $product->company_id)->exists()) {
            abort(403, 'Unauthorized action.');
        }
        $companies = $user->companies()->where('status', 'approved')->orderBy('name')->pluck('name', 'id');
        // TODO: Add other necessary data like categories, product types, etc.
        return view('businessdirectory::frontend.products.edit', compact('product', 'companies'));
    }

    /**
     * Update the specified product in storage.
     */
    public function update(UpdateProductRequest $request, Product $product) // Assuming UpdateProductRequest exists
    {
        $this->authorize('update', $product);
        $validatedData = $request->validated();

        // Handle image uploads if any
        // if ($request->hasFile('featured_image')) { ... }

        $product->update($validatedData);

        return redirect()->route('frontend.businessdirectory.my-products.index')
                         ->with('success', 'Product updated successfully.');
    }

    /**
     * Remove the specified product from storage.
     */
    public function destroy(Product $product)
    {
        $this->authorize('delete', $product);
        // Handle image deletion from storage if any
        $product->delete();
        return redirect()->route('frontend.businessdirectory.my-products.index')
                         ->with('success', 'Product deleted successfully.');
    }

    // Method to manage products within a specific company context (if needed)
    // public function manageCompanyProducts(Company $company)
    // {
    //     $this->authorize('manageProducts', $company); // Policy check on Company
    //     $products = $company->products()->paginate(10);
    //     return view('businessdirectory::frontend.products.manage_company_products', compact('company', 'products'));
    // }
    }
