<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyType;
use Modules\BusinessDirectory\Entities\Review;
use Modules\BusinessDirectory\Http\Requests\StoreCompanyRequest;
use App\Helpers\UserAccessHelper; // Import the helper
use Modules\BusinessDirectory\Http\Requests\UpdateCompanyRequest;
// Assuming you have a way to check subscription features
// use App\Services\SubscriptionService; // Example

class CompanyController extends Controller
{
    use AuthorizesRequests;

    // Note: logoPath and bannerPath properties are removed as paths will be fetched from config.

    public function __construct()
    {
        $this->middleware('auth')->except(['index', 'show', 'indexByType']);
    }

    /**
     * Display a listing of approved companies.
     *
     * @param Request $request
     * @return Renderable
     */
    public function index(Request $request) // This is now ONLY for company listings/search
    {
        $query = Company::with(['companyType', 'user:id,name'])
                        ->withAvg(['reviews' => function ($query) {
                            $query->where('is_approved', true);
                        }], 'rating') // Eager load average rating from approved reviews
                        ->withCount(['reviews' => function ($query) {
                            $query->where('is_approved', true);
                        }]) // Eager load count of approved reviews
                        ->approved() // Assuming Company::scopeApproved()
                        ->orderBy('is_featured', 'desc')
                        ->orderBy('name', 'asc');

        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('city', 'like', "%{$searchTerm}%")
                  ->orWhere('state_province', 'like', "%{$searchTerm}%");
            });
        }

        if ($request->filled('company_type_slug')) {
            $companyType = CompanyType::where('slug', $request->company_type_slug)->first();
            if ($companyType) {
                $query->where('company_type_id', $companyType->id);
            }
        } elseif ($request->filled('company_type_id')) {
             $query->where('company_type_id', $request->company_type_id);
        }

        $companies = $query->paginate(setting('pagination_count', 12))->withQueryString();
        $companyTypes = CompanyType::orderBy('name')->get();
        $pageTitle = __('Company Directory');
        $metaDescription = __('Browse companies in our directory.');

        return view('businessdirectory::frontend.companies.index', compact('companies', 'companyTypes', 'pageTitle', 'metaDescription'));
    }

    /**
     * Display a listing of approved companies filtered by type.
     *
     * @param Request $request
     * @param CompanyType $companyType
     * @return Renderable
     */
    public function indexByType(Request $request, CompanyType $companyType)
    {
        $query = Company::with(['user:id,name'])
                        ->withAvg(['reviews' => function ($query) {
                            $query->where('is_approved', true);
                        }], 'rating') // Eager load average rating from approved reviews
                        ->withCount(['reviews' => function ($query) {
                            $query->where('is_approved', true);
                        }]) // Eager load count of approved reviews
                        ->approved() // Using scope
                        ->where('company_type_id', $companyType->id)
                        ->orderBy('is_featured', 'desc')
                        ->orderBy('name', 'asc');

        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('city', 'like', "%{$searchTerm}%");
            });
        }

        $companies = $query->paginate(setting('pagination_count', 12))->withQueryString();
        $companyTypes = CompanyType::orderBy('name')->get(); // Still useful for a general filter on the page

        return view('businessdirectory::frontend.companies.index', compact('companies', 'companyType', 'companyTypes'));
    }

    /**
     * Display the specified company.
     *
     * @param Company $company
     * @return Renderable
     */
    public function show(Company $company)
    {
        $this->authorize('view', $company);

        $company->load(['companyType', 'user:id,name', 
            'products' => function ($query) {
                $query->where('is_published', true)->orderBy('name');
            }, 
            'reviews' => function ($query) {
                $query->where('is_approved', true)->with('user:id,name')->latest();
            },
            'jobs' => function ($query) { // Eager load open jobs
                $query->where('status', 'open')->latest();
            }, 
            'tenders' => function ($query) { // Eager load open and active tenders
                $query->where('status', 'open')->where('deadline', '>=', now())->latest();
            }
        ]);

        $averageRating = $company->reviews->avg('rating');
        $reviewCount = $company->reviews->count();

        return view('businessdirectory::frontend.companies.show', compact('company', 'averageRating', 'reviewCount'));
    }

    /**
     * Display a listing of companies owned by the authenticated user.
     *
     * @param Request $request
     * @return Renderable
     */
    public function myCompanies(Request $request)
    {
        $user = Auth::user();
        $query = $user->companies()->with('companyType')->orderBy('name');

        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where('name', 'like', "%{$searchTerm}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        if ($request->filled('company_type_id')) {
             $query->where('company_type_id', $request->input('company_type_id'));
        }
        $companies = $query->paginate(10);

        $filterCompanyTypes = CompanyType::orderBy('name')->pluck('name', 'id');

        return view('businessdirectory::frontend.companies.my_companies', compact('companies', 'filterCompanyTypes'));
    }

    /**
     * Show the form for creating a new company.
     *
     * @return Renderable|\Illuminate\Http\RedirectResponse
     */
    public function create()
    {
        $this->authorize('create', Company::class);

        if (!UserAccessHelper::checkLimit(Auth::user(), 'businessdirectory_list_company', 0, false)) {
            return redirect()->route('frontend.businessdirectory.my-companies.index')
                             ->with('error', 'You have reached the maximum limit for creating companies under your current plan. Please upgrade your subscription.');
        }

        $company = new Company();
        $companyTypes = CompanyType::orderBy('name')->pluck('name', 'id');
        return view('businessdirectory::frontend.companies.create', compact('company', 'companyTypes'));
    }

    /**
     * Store a newly created company in storage.
     *
     * @param  \Modules\BusinessDirectory\Http\Requests\StoreCompanyRequest  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(StoreCompanyRequest $request)
    {
        $this->authorize('create', Company::class);

        if (!UserAccessHelper::checkLimitAndDecrement(Auth::user(), 'businessdirectory_list_company')) {
            return redirect()->route('frontend.businessdirectory.my-companies.create')
                             ->with('error', 'You have reached the maximum limit for creating companies. Please upgrade your subscription.')
                             ->withInput();
        }

        $validatedData = $request->validated();

        if ($request->hasFile('logo_path')) {
            $validatedData['logo_path'] = $request->file('logo_path')->store(config('businessdirectory.storage_paths.logos', 'company-logos'), 'public');
        }
        if ($request->hasFile('banner_image_path')) {
            $validatedData['banner_image_path'] = $request->file('banner_image_path')->store(config('businessdirectory.storage_paths.banners', 'company-banners'), 'public');
        }

        $validatedData['user_id'] = Auth::id();
        $validatedData['status'] = setting('auto_approve_companies', false) ? Company::STATUS_APPROVED : Company::STATUS_PENDING_APPROVAL;

        $validatedData = $this->prepareSocialLinks($validatedData);

        Company::create($validatedData);

        return redirect()->route('frontend.businessdirectory.my-companies.index')
                         ->with('success', 'Company listing submitted successfully. It will be reviewed by an administrator.');
    }

    /**
     * Show the form for editing the specified company.
     *
     * @param Company $company
     * @return Renderable
     */
    public function edit(Company $company)
    {
        $this->authorize('update', $company);
        $companyTypes = CompanyType::orderBy('name')->pluck('name', 'id');
        return view('businessdirectory::frontend.companies.edit', compact('company', 'companyTypes'));
    }

    /**
     * Update the specified company in storage.
     *
     * @param  \Modules\BusinessDirectory\Http\Requests\UpdateCompanyRequest  $request
     * @param Company $company
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UpdateCompanyRequest $request, Company $company)
    {
        $this->authorize('update', $company);
        $validatedData = $request->validated();

        if ($request->hasFile('logo_path')) {
            if ($company->logo_path) Storage::disk('public')->delete($company->logo_path);
            $validatedData['logo_path'] = $request->file('logo_path')->store(config('businessdirectory.storage_paths.logos', 'company-logos'), 'public');
        }

        if ($request->hasFile('banner_image_path')) {
            if ($company->banner_image_path) Storage::disk('public')->delete($company->banner_image_path);
            $validatedData['banner_image_path'] = $request->file('banner_image_path')->store(config('businessdirectory.storage_paths.banners', 'company-banners'), 'public');
        }

        // If admin approval is required for changes, reset status
        // if (setting('require_reapproval_on_edit', false)) {
        //    $validatedData['status'] = Company::STATUS_PENDING_APPROVAL;
        // }

        $validatedData = $this->prepareSocialLinks($validatedData);

        $company->update($validatedData);

        return redirect()->route('frontend.businessdirectory.my-companies.index')
                         ->with('success', 'Company updated successfully.');
    }

    /**
     * Remove the specified company from storage.
     *
     * @param Company $company
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Company $company)
    {
        $this->authorize('delete', $company);

        if ($company->logo_path) Storage::disk('public')->delete($company->logo_path);
        if ($company->banner_image_path) Storage::disk('public')->delete($company->banner_image_path);

        // Before deletion, consider implications for related items:
        // - Products, Jobs, Tenders: Should they be deleted, unassigned, or prevent company deletion?
        // Or rely on database cascade constraints

        $company->delete(); // Soft delete

        return redirect()->route('frontend.businessdirectory.my-companies.index')
                         ->with('success', 'Company deleted successfully.');
    }

    /**
     * Prepare social links array from individual URL fields.
     *
     * @param array $validatedData
     * @return array
     */
    private function prepareSocialLinks(array $validatedData): array
    {
        $socialLinks = [
            'facebook' => $validatedData['facebook_url'] ?? null,
            'twitter' => $validatedData['twitter_url'] ?? null,
            'linkedin' => $validatedData['linkedin_url'] ?? null,
            'instagram' => $validatedData['instagram_url'] ?? null,
            // Add other platforms if you have them in your form (e.g., youtube_url)
        ];
        $validatedData['social_links'] = array_filter($socialLinks); // Store only non-empty links

        // Remove individual social link keys as they are not direct columns in the 'bd_companies' table
        unset(
            $validatedData['facebook_url'], 
            $validatedData['twitter_url'], 
            $validatedData['linkedin_url'], 
            $validatedData['instagram_url']
            // Unset other individual platform URLs if added above
        );
        return $validatedData;
    }
}
