<?php

namespace Modules\BusinessDirectory\Http\Controllers\Frontend;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Modules\BusinessDirectory\Entities\Bid;
use App\Helpers\UserAccessHelper; // Import the helper
use Modules\BusinessDirectory\Http\Requests\StoreBidRequest;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Company;
// Assuming you have notifications set up or plan to
// use Modules\BusinessDirectory\Notifications\NewBidSubmittedNotification;
// use Modules\BusinessDirectory\Notifications\BidConfirmationNotification;
// use Modules\BusinessDirectory\Notifications\BidWithdrawnNotification;

class BidController extends Controller
{
    use AuthorizesRequests;

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Store a newly created bid in storage.
     *
     * @param  \Modules\BusinessDirectory\Http\Requests\StoreBidRequest  $request
     * @param  \Modules\BusinessDirectory\Entities\Tender  $tender
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(StoreBidRequest $request, Tender $tender)
    {
        $user = Auth::user();

        // Authorize creating a bid for this tender
        $this->authorize('create', [Bid::class, $tender]);

        if (!UserAccessHelper::checkLimitAndDecrement(Auth::user(), 'businessdirectory_bid_on_tenders')) {
            return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                             ->with('error', 'You have reached the bid submission limit for your current plan.');
        }


        if ($tender->status !== 'open' || ($tender->deadline && $tender->deadline->isPast())) {
            return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                             ->with('error', 'This tender is no longer accepting bids.');
        }

        if ($tender->company && $tender->company->user_id === $user->id) {
             return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                             ->with('error', 'You cannot bid on your own tender.');
        }

        $existingBid = $tender->bids()
            ->where(function ($query) use ($user) {
                $query->where('bidder_user_id', $user->id)
                      ->orWhereIn('bidder_company_id', $user->companies()->pluck('id')->toArray());
            })
            ->exists();

        if ($existingBid) {
            return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                             ->with('error', 'You have already submitted a bid for this tender.');
        }

        $validated = $request->validated();

        $bidData = [
            'tender_id' => $tender->id,
            'proposal_details' => $validated['proposal_details'],
            'bid_amount' => $validated['bid_amount'] ?? null,
            // 'attachments' => [], // Handle attachments if implemented
            'status' => 'submitted',
            'is_flagged' => false,
        ];

        if ($request->input('bid_as') === 'user') {
            $bidData['bidder_user_id'] = $user->id;
            $bidData['bidder_company_id'] = null;
        } elseif ($request->input('bid_as') === 'company') {
            $company = Company::where('id', $validated['bidder_company_id_input'])
                              ->where('user_id', $user->id)
                              ->first();

            if (!$company) {
                return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                                 ->with('error', 'Invalid company selected or you do not own this company.')->withInput();
            }
            $bidData['bidder_company_id'] = $company->id;
            $bidData['bidder_user_id'] = $user->id;
        } else {
            return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                             ->with('error', 'Invalid bid submission type.')->withInput();
        }

        $bid = Bid::create($bidData);

        // Notify Tender Owner about the new bid
        if ($tender->company && $tender->company->owner) { // Good to keep the check
            event(new \Modules\BusinessDirectory\Events\NewBidSubmitted($bid));
        }

        // Notify Bidder (Confirmation)
        event(new \Modules\BusinessDirectory\Events\BidSubmissionConfirmed($bid, $user));

        return redirect()->route('frontend.businessdirectory.tenders.show', $tender->slug)
                         ->with('success', 'Your bid has been submitted successfully!');
    }

    /**
     * Display a listing of the bids submitted by the authenticated user.
     *
     * @param Request $request
     * @return Renderable
     */
    public function mySubmittedBids(Request $request)
    {
        $user = Auth::user();
        $userCompanyIds = $user->companies()->pluck('id')->toArray();

        $query = Bid::where('bidder_user_id', $user->id)
                    ->orWhereIn('bidder_company_id', $userCompanyIds);

        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->whereHas('tender', function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        $bids = $query
                    ->with(['tender:id,title,slug,budget_currency', 'bidderCompany:id,name'])
                    ->orderBy('created_at', 'desc')
                    ->paginate(15);

        return view('businessdirectory::frontend.bids.my_submitted_bids', compact('bids'));
    }

    /**
     * Display the specified bid submitted by the authenticated user.
     *
     * @param  \Modules\BusinessDirectory\Entities\Bid  $bid
     * @return Renderable
     */
    public function showMyBid(Bid $bid)
    {
        $this->authorize('view', $bid);

        $bid->load(['tender.company', 'bidderCompany', 'bidderUser']);

        return view('businessdirectory::frontend.bids.show_my_bid', compact('bid'));
    }

    /**
     * Withdraw the specified bid submitted by the authenticated user.
     *
     * @param  \Modules\BusinessDirectory\Entities\Bid  $bid
     * @return \Illuminate\Http\RedirectResponse
     */
    public function withdrawBid(Bid $bid)
    {
        $this->authorize('withdraw', $bid);

        if ($bid->status === 'accepted' || $bid->status === 'rejected' || $bid->status === 'withdrawn') {
            return redirect()->route('frontend.businessdirectory.my-submitted-bids.index')
                             ->with('error', 'This bid cannot be withdrawn as it has already been processed or withdrawn.');
        }

        $bid->update(['status' => 'withdrawn']);

        // Optionally, notify the tender owner about the withdrawal
        if ($bid->tender && $bid->tender->company && $bid->tender->company->owner) { // Good to keep the check
            event(new \Modules\BusinessDirectory\Events\BidWithdrawn($bid));
        }

        return redirect()->route('frontend.businessdirectory.my-submitted-bids.index')->with('success', 'Your bid has been successfully withdrawn.');
    }
}
