<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\Tender;
use Modules\BusinessDirectory\Entities\Bid; // Import Bid model
use Modules\BusinessDirectory\Http\Requests\Admin\StoreTenderRequest;
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateTenderRequest;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
// use Modules\BusinessDirectory\Entities\CompanyType; // Not directly used in this controller
// use Modules\BusinessDirectory\Entities\Product; // Not directly used in this controller

class TenderController extends Controller
{
    public function __construct()
    {
        // Policy authorization per method.
    }

    public function index(Request $request)
    {
        $this->authorize('viewAny', Tender::class);
        $query = Tender::with('company')->withCount('bids')->orderBy('created_at', 'desc');

        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhereHas('company', function ($companyQuery) use ($searchTerm) {
                      $companyQuery->where('name', 'like', "%{$searchTerm}%");
                  });
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        $tenders = $query->paginate(15)->withQueryString(); // Added withQueryString
        $statuses = Tender::getStatuses(); // Assuming Tender model has getStatuses()
        $companies = Company::orderBy('name')->pluck('name', 'id'); // For the filter dropdown

        return view('businessdirectory::admin.tenders.index', compact('tenders', 'statuses', 'companies'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorize('create', Tender::class);
        $tender = new Tender(); // For form model binding
        $companies = Company::orderBy('name')->pluck('name', 'id');
        $statuses = Tender::getStatuses();
        return view('businessdirectory::admin.tenders.create', compact('tender', 'companies', 'statuses'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreTenderRequest $request)
    {
        $this->authorize('create', Tender::class);
        $validatedData = $request->validated();
        // Slug is handled by HasSlug trait in the model

        Tender::create($validatedData);
        return redirect()->route('admin.businessdirectory.tenders.index')
                         ->with('success', 'Tender created successfully.');
    }

    public function show(Tender $tender)
    {
        $this->authorize('view', $tender);
        // Typically, for admin, edit is more common than show.
        // You might want to load bids here or redirect to edit.
        // For now, redirecting to edit or showing a simple view.
        // Let's assume we want to show bids on the show page for admin
        return $this->showBids($tender);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Tender $tender)
    {
        $this->authorize('update', $tender);
        $companies = Company::orderBy('name')->pluck('name', 'id');
        $statuses = Tender::getStatuses();
        return view('businessdirectory::admin.tenders.edit', compact('tender', 'companies', 'statuses'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateTenderRequest $request, Tender $tender)
    {
        $this->authorize('update', $tender);
        $validatedData = $request->validated();
        // Slug is handled by HasSlug trait if title changes

        $tender->update($validatedData);
        return redirect()->route('admin.businessdirectory.tenders.index')
                         ->with('success', 'Tender updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Tender $tender)
    {
        $this->authorize('delete', $tender);
        // Consider what happens to bids associated with this tender.
        // By default, if bids have a foreign key with onDelete('cascade'), they will be deleted.
        $tender->delete();
        return redirect()->route('admin.businessdirectory.tenders.index')
                         ->with('success', 'Tender deleted successfully.');
    }

    /**
     * Display the bids for a specific tender.
     */
    public function showBids(Tender $tender)
    {
        $this->authorize('view', $tender); // Or a specific 'viewBids' permission
        $tender->load('company'); // Ensure company is loaded
        $bids = $tender->bids()
                       ->with(['bidderCompany.owner', 'bidderUser']) // Eager load bidderCompany, its owner, and bidderUser
                       ->orderBy('created_at', 'desc')
                       ->paginate(10);

        return view('businessdirectory::admin.tenders.show_bids', compact('tender', 'bids'));
    }

    /**
     * Toggle the flagged status of a bid.
     */
    public function toggleBidFlag(Bid $bid)
    {
        $this->authorize('update', $bid->tender); // Admin can update bids related to a tender
        $bid->update(['is_flagged' => !$bid->is_flagged]);

        $message = $bid->is_flagged ? 'Bid has been flagged for review.' : 'Bid has been unflagged.';

        return redirect()->back()->with('success', $message);
    }
}
