<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\BusinessDirectory\Entities\Order;
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateOrderRequest; // Keep if used

class OrderController extends Controller
{
    public function __construct()
    {
        // Policy authorization per method.
    }
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', Order::class);
        $query = Order::with(['user', 'company'])->latest();

        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where('order_number', 'like', "%{$searchTerm}%")
                  ->orWhereHas('user', fn($q) => $q->where('name', 'like', "%{$searchTerm}%")->orWhere('email', 'like', "%{$searchTerm}%"))
                  ->orWhereHas('company', fn($q) => $q->where('name', 'like', "%{$searchTerm}%"));
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        $orders = $query->paginate(15)->withQueryString();

        // Use the static methods from the Order model if they exist, or config values
        $order_statuses = method_exists(Order::class, 'getStatuses') ? Order::getStatuses() : config('businessdirectory.order_statuses', ['pending' => 'Pending', 'processing' => 'Processing', 'shipped' => 'Shipped', 'delivered' => 'Delivered', 'completed' => 'Completed', 'cancelled' => 'Cancelled', 'refunded' => 'Refunded', 'payment_failed' => 'Payment Failed']);
        $payment_statuses = method_exists(Order::class, 'getPaymentStatuses') ? Order::getPaymentStatuses() : config('businessdirectory.payment_statuses', ['pending' => 'Pending', 'paid' => 'Paid', 'failed' => 'Failed', 'refunded' => 'Refunded']);

        return view('businessdirectory::admin.orders.index', compact('orders', 'order_statuses', 'payment_statuses'));
    }

    /**
     * Show the specified resource.
     * @param Order $order
     * @return Renderable
     */
    public function show(Order $order)
    {
        $this->authorize('view', $order);
        $order->load(['user', 'company', 'items.product']);
        return view('businessdirectory::admin.orders.show', compact('order'));
    }

    public function edit(Order $order)
    {
        $this->authorize('update', $order);
        // Load necessary data for the edit form (e.g., statuses)
        $order_statuses = Order::getStatuses();
        $payment_statuses = Order::getPaymentStatuses();
        return view('businessdirectory::admin.orders.edit', compact('order', 'order_statuses', 'payment_statuses'));
    }

    /**
     * Update the specified resource in storage. (e.g., order status, payment status)
     * @param Request $request
     * @param Order $order
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UpdateOrderRequest $request, Order $order)
    {
        $this->authorize('update', $order);
        $validatedData = $request->validated();

        $order->update($validatedData);

        // Notify user about order status change
        // OrderStatusUpdatedNotification::dispatch($order->user, $order);
        // if ($order->wasChanged('status') || $order->wasChanged('payment_status')) {
            // Send notification
        // }

        return back()->with('success', 'Order updated successfully.');
    }

    // No destroy method for orders usually, they are marked as cancelled/refunded.
}