<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // Import the trait
use Modules\BusinessDirectory\Entities\Job;
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\JobApplication;
use Modules\BusinessDirectory\Http\Requests\Admin\StoreJobRequest; // Keep if used
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateJobRequest; // Keep if used

class JobController extends Controller
{
    use AuthorizesRequests; // Use the trait

    public function __construct()
    {
        // Policy authorization per method.
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', Job::class);

        $query = Job::with('company')->latest();

        if ($request->filled('search_title')) {
            $query->where('title', 'like', '%' . $request->search_title . '%');
        }

        if ($request->filled('job_status_filter')) {
            $query->where('status', $request->job_status_filter);
        }
        // Add other filters like company_id if needed

        $jobs = $query->paginate(15)->withQueryString();
        $job_statuses_filter = Job::getStatuses();
        // $companies_filter = Company::orderBy('name')->pluck('name', 'id'); // If you add company filter

        return view('businessdirectory::admin.jobs.index', compact('jobs', 'job_statuses_filter' /*, 'companies_filter' */));
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        $this->authorize('create', Job::class);
        $companies = Company::where('status', 'approved')->pluck('name', 'id');
        $job_statuses = Job::getStatuses();
        $job_types = Job::getJobTypes();
        $categories = \Modules\BusinessDirectory\Entities\Category::orderBy('name')->pluck('name', 'id'); // Assuming you have categories for jobs
        return view('businessdirectory::admin.jobs.create', compact('companies', 'job_statuses', 'job_types', 'categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreJobRequest $request)
    {
        $this->authorize('create', Job::class);
        $validatedData = $request->validated();
        // Slug is handled by HasSlug trait in the model

        // Admin can assign user_id, otherwise default to company owner or logged-in admin
        if (empty($validatedData['user_id']) && !empty($validatedData['company_id'])) {
            $company = Company::find($validatedData['company_id']);
            $validatedData['user_id'] = $company->user_id ?? auth()->id();
        } elseif (empty($validatedData['user_id'])) {
            $validatedData['user_id'] = auth()->id();
        }

        Job::create($validatedData);

        return redirect()->route('admin.businessdirectory.jobs.index')
            ->with('success', 'Job created successfully.');
    }

    /**
     * Show the specified resource.
     */
    public function show(Job $job)
    {
        $this->authorize('view', $job);
        $job->load('company', 'jobApplications.user');
        return view('businessdirectory::admin.jobs.show', compact('job'));
    }

    /**
     * Show the form for editing the specified resource.
     * @param Job $job
     */
    public function edit(Job $job)
    {
        $this->authorize('update', $job);
        $companies = Company::where('status', 'approved')->pluck('name', 'id');
        $job_statuses = Job::getStatuses();
        $job_types = Job::getJobTypes();
        $categories = \Modules\BusinessDirectory\Entities\Category::orderBy('name')->pluck('name', 'id');
        return view('businessdirectory::admin.jobs.edit', compact('job', 'companies', 'job_statuses', 'job_types', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateJobRequest $request, Job $job)
    {
        $this->authorize('update', $job);
        $validatedData = $request->validated();
        // Slug is handled by HasSlug trait if title changes

        $job->update($validatedData);

        return redirect()->route('admin.businessdirectory.jobs.index')
            ->with('success', 'Job updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Job $job)
    {
        $this->authorize('delete', $job);
        try {
            // Consider what happens to job applications.
            // If JobApplication model has a foreign key with onDelete('cascade'), they will be deleted.
            // Otherwise, you might want to delete them manually or prevent deletion if applications exist.
            // $job->jobApplications()->delete(); // Example

            $job->delete();
            return redirect()->route('admin.businessdirectory.jobs.index')->with('success', 'Job deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.businessdirectory.jobs.index')->with('error', 'Failed to delete job. It might have related data.');
        }
    }

    /**
     * Update the status of a job application.
     */
    public function updateApplicationStatus(Request $request, JobApplication $application)
    {
        $this->authorize('update', $application->job); // Admin can update applications for any job
        $request->validate(['status' => 'required|in:applied,shortlisted,interviewed,rejected,hired,withdrawn']);
        $application->update(['status' => $request->status]);
        return back()->with('success', 'Application status updated.');
    }
}