<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\BusinessDirectory\Entities\CompanyType;
use Modules\BusinessDirectory\Http\Requests\Admin\StoreCompanyTypeRequest;
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateCompanyTypeRequest;
use Illuminate\Support\Facades\Storage;

class CompanyTypeController extends Controller
{
    public function __construct()
    {
        // Assuming admin routes are protected. Policy authorization per method.
    }

    public function index(Request $request)
    {
        $this->authorize('viewAny', CompanyType::class);
        $query = CompanyType::query()->latest();

        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        $companyTypes = $query->paginate(15);
        return view('businessdirectory::admin.company_types.index', compact('companyTypes'));
    }

    public function create()
    {
        $this->authorize('create', CompanyType::class);
        return view('businessdirectory::admin.company_types.create');
    }

    public function store(StoreCompanyTypeRequest $request)
    {
        $this->authorize('create', CompanyType::class);
        $validatedData = $request->validated();

        if ($request->hasFile('icon')) {
            $validatedData['icon'] = $request->file('icon')->store('company_type_icons', 'public');
        }

        CompanyType::create($validatedData);

        return redirect()->route('admin.businessdirectory.company-types.index')
                         ->with('success', 'Company Type created successfully.');
    }

    public function edit(CompanyType $companyType)
    {
        $this->authorize('update', $companyType);
        return view('businessdirectory::admin.company_types.edit', compact('companyType'));
    }

    public function update(UpdateCompanyTypeRequest $request, CompanyType $companyType)
    {
        $this->authorize('update', $companyType);
        $validatedData = $request->validated();

        if ($request->hasFile('icon')) {
            if ($companyType->icon) {
                Storage::disk('public')->delete($companyType->icon);
            }
            $validatedData['icon'] = $request->file('icon')->store('company_type_icons', 'public');
        }

        $companyType->update($validatedData);

        return redirect()->route('admin.businessdirectory.company-types.index')
                         ->with('success', 'Company Type updated successfully.');
    }

    public function destroy(CompanyType $companyType)
    {
        $this->authorize('delete', $companyType);

        if ($companyType->companies()->count() > 0) {
            return redirect()->route('admin.businessdirectory.company-types.index')
                             ->with('error', 'Cannot delete company type. It is associated with existing companies.');
        }

        if ($companyType->icon) {
            Storage::disk('public')->delete($companyType->icon);
        }
        $companyType->delete();

        return redirect()->route('admin.businessdirectory.company-types.index')
                         ->with('success', 'Company Type deleted successfully.');
    }
}