<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Modules\BusinessDirectory\Entities\CompanyTask;
use App\Entities\User; // Assuming global User model

class CompanyTaskController extends Controller
{
    // Note: Admin task management is often done via the project's show page.
    // These methods provide direct CRUD if needed, but might be simplified.

    /**
     * Show the form for creating a new task for a project.
     * @param CompanyProject $project
     * @return Renderable
     */
    public function create(CompanyProject $project)
    {
        $assignees = User::orderBy('name')->pluck('name', 'id'); // Or filter by users related to the company
        $task_statuses = ['todo' => 'To Do', 'in_progress' => 'In Progress', 'review' => 'Review', 'completed' => 'Completed', 'blocked' => 'Blocked'];
        $priorities = ['low' => 'Low', 'medium' => 'Medium', 'high' => 'High', 'urgent' => 'Urgent'];
        return view('businessdirectory::admin.tasks.create', compact('project', 'assignees', 'task_statuses', 'priorities'));
    }

    /**
     * Store a newly created task in storage.
     * @param Request $request
     * @param CompanyProject $project
     * @return Renderable
     */
    public function store(Request $request, CompanyProject $project)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assignee_user_id' => 'nullable|exists:users,id',
            'due_date' => 'nullable|date',
            'priority' => 'nullable|string|max:50',
            'status' => 'required|in:todo,in_progress,review,completed,blocked',
        ]);

        $taskData = $request->all();
        $taskData['project_id'] = $project->id;
        CompanyTask::create($taskData);

        return redirect()->route('admin.businessdirectory.projects.show', $project->id)
            ->with('success', 'Task created successfully.');
    }

    /**
     * Show the form for editing the specified task.
     * @param CompanyProject $project (passed for context, though task is route model bound)
     * @param CompanyTask $task
     * @return Renderable
     */
    public function edit(CompanyProject $project, CompanyTask $task)
    {
        if ($task->project_id !== $project->id) {
            abort(404); // Task does not belong to this project
        }
        $assignees = User::orderBy('name')->pluck('name', 'id');
        $task_statuses = ['todo' => 'To Do', 'in_progress' => 'In Progress', 'review' => 'Review', 'completed' => 'Completed', 'blocked' => 'Blocked'];
        $priorities = ['low' => 'Low', 'medium' => 'Medium', 'high' => 'High', 'urgent' => 'Urgent'];
        return view('businessdirectory::admin.tasks.edit', compact('project', 'task', 'assignees', 'task_statuses', 'priorities'));
    }

    /**
     * Update the specified task in storage.
     * @param Request $request
     * @param CompanyProject $project
     * @param CompanyTask $task
     * @return Renderable
     */
    public function update(Request $request, CompanyProject $project, CompanyTask $task)
    {
        if ($task->project_id !== $project->id) {
            abort(404);
        }
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assignee_user_id' => 'nullable|exists:users,id',
            'due_date' => 'nullable|date',
            'priority' => 'nullable|string|max:50',
            'status' => 'required|in:todo,in_progress,review,completed,blocked',
        ]);

        $task->update($request->all());

        return redirect()->route('admin.businessdirectory.projects.show', $project->id)
            ->with('success', 'Task updated successfully.');
    }

    /**
     * Remove the specified task from storage.
     * @param CompanyProject $project
     * @param CompanyTask $task
     * @return Renderable
     */
    public function destroy(CompanyProject $project, CompanyTask $task)
    {
        if ($task->project_id !== $project->id) {
            return response()->json(['error' => 'Task does not belong to this project.'], 403);
        }
        try {
            $task->delete();
            // If request is AJAX, return JSON, otherwise redirect
            if (request()->ajax()) {
                return response()->json(['success' => 'Task deleted successfully.']);
            }
            return redirect()->route('admin.businessdirectory.projects.show', $project->id)
                             ->with('success', 'Task deleted successfully.');
        } catch (\Exception $e) {
            if (request()->ajax()) {
                return response()->json(['error' => 'Failed to delete task.'], 500);
            }
            return redirect()->route('admin.businessdirectory.projects.show', $project->id)
                             ->with('error', 'Failed to delete task.');
        }
    }
}