<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\BusinessDirectory\Entities\CompanyProject;
use Modules\BusinessDirectory\Entities\Company; // Keep if needed for create/edit forms

class CompanyProjectController extends Controller
{
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', CompanyProject::class); // Assuming you have a policy

        $query = CompanyProject::with('company')->latest();

        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where('name', 'like', "%{$searchTerm}%")
                  ->orWhereHas('company', function ($q) use ($searchTerm) {
                      $q->where('name', 'like', "%{$searchTerm}%");
                  });
        }

        if ($request->filled('status_filter')) {
            $query->where('status', $request->status_filter);
        }

        $projects = $query->paginate(15)->withQueryString();
        $project_statuses_filter = CompanyProject::getStatuses();

        return view('businessdirectory::admin.projects.index', compact('projects', 'project_statuses_filter'));
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        $companies = Company::where('status', 'approved')->orderBy('name')->pluck('name', 'id');
        $project_statuses = CompanyProject::getStatuses();
        return view('businessdirectory::admin.projects.create', compact('companies', 'project_statuses'));
    }

    /**
     * Store a newly created resource in storage.
     * @param Request $request
     * @return Renderable
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:bd_companies,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'budget' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|max:10',
            'status' => 'required|in:planning,in_progress,completed,on_hold,cancelled,draft',
            'client_company_id' => 'nullable|exists:bd_companies,id',
        ]);

        CompanyProject::create($request->all());

        return redirect()->route('admin.businessdirectory.projects.index')
            ->with('success', 'Project created successfully.');
    }

    /**
     * Show the specified resource (tasks for this project).
     * @param CompanyProject $project
     * @return Renderable
     */
    public function show(CompanyProject $project)
    {
        $project->load('company', 'tasks.assignee'); // Eager load tasks and their assignees
        return view('businessdirectory::admin.projects.show', compact('project'));
    }

    /**
     * Show the form for editing the specified resource.
     * @param CompanyProject $project
     * @return Renderable
     */
    public function edit(CompanyProject $project)
    {
        $companies = Company::where('status', 'approved')->orderBy('name')->pluck('name', 'id');
        $project_statuses = CompanyProject::getStatuses();
        return view('businessdirectory::admin.projects.edit', compact('project', 'companies', 'project_statuses'));
    }

    /**
     * Update the specified resource in storage.
     * @param Request $request
     * @param CompanyProject $project
     * @return Renderable
     */
    public function update(Request $request, CompanyProject $project)
    {
        $request->validate([
            'company_id' => 'required|exists:bd_companies,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'budget' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|max:10',
            'status' => 'required|in:planning,in_progress,completed,on_hold,cancelled,draft',
            'client_company_id' => 'nullable|exists:bd_companies,id',
        ]);

        $project->update($request->all());

        return redirect()->route('admin.businessdirectory.projects.index')
            ->with('success', 'Project updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     * @param CompanyProject $project
     * @return Renderable
     */
    public function destroy(CompanyProject $project)
    {
        try {
            // Consider deleting related tasks or handling them based on business logic
            $project->tasks()->delete(); // Example: delete related tasks
            // Add authorization check for deleting
            $this->authorize('delete', $project);
            $project->delete();
            return redirect()->route('admin.businessdirectory.projects.index')->with('success', 'Project and its tasks deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.businessdirectory.projects.index')->with('error', 'Failed to delete project: ' . $e->getMessage());
        }
    }
}