<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
// Adjust namespace if your Entities are in Entities
use Modules\BusinessDirectory\Entities\Company;
use Modules\BusinessDirectory\Entities\CompanyType;
use Modules\BusinessDirectory\Entities\Category; // Import Category model
use App\Models\User; // Standardized User model path
use Illuminate\Support\Facades\Storage;
use Modules\BusinessDirectory\Http\Requests\Admin\StoreCompanyRequest as AdminStoreCompanyRequest;
use Modules\BusinessDirectory\Http\Requests\Admin\UpdateCompanyRequest as AdminUpdateCompanyRequest;

class CompanyController extends Controller
{
    public function __construct()
    {
        // Assuming admin routes are already protected by an admin auth middleware
        // Policy authorization will be done per method.
        // $this->authorizeResource(Company::class, 'company'); // Alternative for full resource controller
        // $this->middleware('can:manage-companies'); // Example permission
    }

    public function index(Request $request)
    {
        $query = Company::with('user', 'companyType')->latest();

        // Admin can view all, so typically no specific viewAny policy check here for index listing
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%")
                  ->orWhereHas('user', function ($userQuery) use ($searchTerm) {
                      $userQuery->where('name', 'like', "%{$searchTerm}%")
                                ->orWhere('email', 'like', "%{$searchTerm}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('company_type_id')) {
            $query->where('company_type_id', $request->company_type_id);
        }

        $companies = $query->paginate(15);
        $companyTypes = CompanyType::orderBy('name')->pluck('name', 'id');
        $statuses = Company::getStatuses(); // Using the static method from Company model

        return view('businessdirectory::admin.companies.index', compact('companies', 'companyTypes', 'statuses'));
    }

    public function create()
    {
        $this->authorize('create', Company::class); // Authorize creation
        $companyTypes = CompanyType::orderBy('name')->pluck('name', 'id');
        $users = User::orderBy('name')->pluck('name', 'id'); // For assigning a user if admin creates
        $statuses = Company::getStatuses();
        $categories = Category::orderBy('name')->pluck('name', 'id'); // Fetch categories
        return view('businessdirectory::admin.companies.create', compact('companyTypes', 'users', 'statuses', 'categories'));
    }

    public function store(AdminStoreCompanyRequest $request)
    {
        $this->authorize('create', Company::class); // Authorize creation
        $validatedData = $request->validated();

        $companyData = $validatedData;
        // Slug is handled by HasSlug trait in the model on saving

        if ($request->hasFile('logo')) {
            $companyData['logo'] = $request->file('logo')->store('company_logos', 'public');
        }
        if ($request->hasFile('banner')) {
            $companyData['banner'] = $request->file('banner')->store('company_banners', 'public');
        }

        // Handle status-specific timestamps
        if ($request->input('status') === 'approved') {
            $companyData['approved_at'] = now();
        }

        // Consolidate social links
        $companyData['social_links'] = [
            'facebook' => $request->input('social_links.facebook'),
            'twitter' => $request->input('social_links.twitter'),
            'linkedin' => $request->input('social_links.linkedin'),
            'instagram' => $request->input('social_links.instagram'),
        ];
        $companyData['social_links'] = array_filter($companyData['social_links']); // Remove empty values

        // Boolean fields are handled by the FormRequest's prepareForValidation or directly from validated data if set up correctly
        $companyData['is_featured'] = $request->boolean('is_featured');
        $companyData['is_verified'] = $request->boolean('is_verified');
        $companyData['is_ecommerce'] = $request->boolean('is_ecommerce');

        Company::create($companyData);

        return redirect()->route('admin.businessdirectory.companies.index')
                         ->with('success', 'Company created successfully.');
    }

    public function show(Company $company)
    {
        $this->authorize('view', $company); // Authorize viewing this specific company
        $company->load('user', 'companyType', 'products', 'tenders'); // Eager load relations
        return view('businessdirectory::admin.companies.show', compact('company'));
    }

    public function edit(Company $company)
    {
        $this->authorize('update', $company); // Authorize updating this specific company
        $companyTypes = CompanyType::orderBy('name')->pluck('name', 'id');
        $users = User::orderBy('name')->pluck('name', 'id');
        $statuses = Company::getStatuses();
        $categories = Category::orderBy('name')->pluck('name', 'id'); // Fetch categories
        return view('businessdirectory::admin.companies.edit', compact('company', 'companyTypes', 'users', 'statuses', 'categories'));
    }

    public function update(AdminUpdateCompanyRequest $request, Company $company)
    {
        $this->authorize('update', $company); // Authorize updating this specific company
        $validatedData = $request->validated();

        $companyData = $validatedData;
        // The HasSlug trait in the Company model handles slug updates automatically if 'name' changes.

        if ($request->hasFile('logo')) {
            if ($company->logo) Storage::disk('public')->delete($company->logo);
            $companyData['logo'] = $request->file('logo')->store('company_logos', 'public');
        }
        // Ensure the form field name for banner is 'banner'
        if ($request->hasFile('banner')) {
            if ($company->banner) Storage::disk('public')->delete($company->banner);
            $companyData['banner'] = $request->file('banner')->store('company_banners', 'public');
        }

        // Handle status changes and timestamps
        if ($company->status !== $request->input('status')) {
            $companyData['approved_at'] = ($request->input('status') === 'approved') ? now() : null;
            $companyData['rejected_at'] = ($request->input('status') === 'rejected') ? now() : null;
            $companyData['rejection_reason'] = ($request->input('status') === 'rejected') ? $request->input('rejection_reason', $company->rejection_reason) : null;
            $companyData['suspended_at'] = ($request->input('status') === 'suspended') ? now() : null;
            $companyData['suspension_reason'] = ($request->input('status') === 'suspended') ? $request->input('suspension_reason', $company->suspension_reason) : null;
        } else {
            // If status hasn't changed, but reasons might have been submitted (e.g. editing a rejected company)
            $companyData['rejection_reason'] = ($request->input('status') === 'rejected') ? $request->input('rejection_reason', $company->rejection_reason) : null;
            $companyData['suspension_reason'] = ($request->input('status') === 'suspended') ? $request->input('suspension_reason', $company->suspension_reason) : null;
        }

        // Ensure boolean fields are correctly set
        $companyData['is_featured'] = $request->boolean('is_featured');
        $companyData['is_verified'] = $request->boolean('is_verified');
        $companyData['is_ecommerce'] = $request->boolean('is_ecommerce');

        // Consolidate social links
        $companyData['social_links'] = [
            'facebook' => $request->input('social_links.facebook'),
            'twitter' => $request->input('social_links.twitter'),
            'linkedin' => $request->input('social_links.linkedin'),
            'instagram' => $request->input('social_links.instagram'),
        ];
        $companyData['social_links'] = array_filter($companyData['social_links']);

        $company->update($companyData);


        return redirect()->route('admin.businessdirectory.companies.index')
                         ->with('success', 'Company updated successfully.');
    }

    public function destroy(Company $company)
    {
        $this->authorize('delete', $company); // Authorize deleting this specific company
        if ($company->logo) Storage::disk('public')->delete($company->logo);
        if ($company->banner) Storage::disk('public')->delete($company->banner);
        // Consider soft delete implications or related data (products, tenders)
        $company->delete(); // This will soft delete if SoftDeletes trait is used

        return redirect()->route('admin.businessdirectory.companies.index')
                         ->with('success', 'Company deleted successfully.');
    }

    public function approve(Company $company)
    {
        $this->authorize('update', $company); // Or a more specific 'approve' policy method
        $company->update([
            'status' => 'approved',
            'approved_at' => now(),
            'rejected_at' => null,
            'rejection_reason' => null,
            'suspended_at' => null,
            'suspension_reason' => null,
        ]);
        // Optionally, send notification to user
        return redirect()->back()->with('success', 'Company approved successfully.');
    }

    public function reject(Request $request, Company $company)
    {
        $this->authorize('update', $company); // Or a more specific 'reject' policy method
        $request->validate(['rejection_reason' => 'required|string|max:1000']);
        $company->update([
            'status' => 'rejected',
            'rejected_at' => now(),
            'rejection_reason' => $request->rejection_reason,
            'approved_at' => null,
            'suspended_at' => null,
            'suspension_reason' => null,
        ]);
        // Optionally, send notification to user
        return redirect()->back()->with('success', 'Company rejected successfully.');
    }

    public function suspend(Request $request, Company $company)
    {
        $this->authorize('update', $company); // Or a more specific 'suspend' policy method
        $request->validate(['suspension_reason' => 'required|string|max:1000']);
        $company->update([
            'status' => 'suspended',
            'suspended_at' => now(),
            'suspension_reason' => $request->suspension_reason,
            // 'approved_at' => null, // Keep approved_at if it was previously approved
        ]);
        // Optionally, send notification to user
        return redirect()->back()->with('success', 'Company suspended successfully.');
    }

    public function reactivate(Company $company)
    {
        $this->authorize('update', $company); // Or a more specific 'reactivate' policy method
        // Reactivate typically means setting back to 'approved' or 'pending'
        // depending on your workflow. Let's assume 'approved'.
        $company->update([
            'status' => 'approved', // Or 'pending' if it needs re-review
            'approved_at' => $company->approved_at ?? now(), // Keep original or set new
            'rejected_at' => null,
            'rejection_reason' => null,
            'suspended_at' => null,
            'suspension_reason' => null,
        ]);
        return redirect()->back()->with('success', 'Company reactivated successfully.');
    }

    public function toggleFeatured(Company $company)
    {
        $this->authorize('update', $company);
        $company->is_featured = !$company->is_featured;
        $company->save();

        return redirect()->back()->with('success', 'Company featured status updated.');
    }

    public function toggleVerified(Company $company)
    {
        $this->authorize('update', $company);
        $company->is_verified = !$company->is_verified;
        $company->save();

        return redirect()->back()->with('success', 'Company verified status updated.');
    }


}
