<?php

namespace Modules\BusinessDirectory\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Modules\BusinessDirectory\Settings\BusinessDirectorySettings;

class BusinessDirectorySettingController extends Controller
{
    public function __construct()
    {
        // Add any necessary middleware, e.g., for admin authentication
        // $this->middleware('auth:admin'); // Example
    }

    /**
     * Show the form for editing the settings.
     *
     * @param BusinessDirectorySettings $settings
     * @return \Illuminate\Contracts\View\View
     */
    public function edit(BusinessDirectorySettings $settings)
    {
        $this->authorize('manage_settings', BusinessDirectorySettings::class); // Assuming a policy or gate
        return view('businessdirectory::admin.settings.edit', compact('settings'));
    }

    /**
     * Update the settings in storage.
     *
     * @param Request $request
     * @param BusinessDirectorySettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, BusinessDirectorySettings $settings)
    {
        $this->authorize('manage_settings', BusinessDirectorySettings::class);

        $validatedData = $request->validate([
            'site_name' => 'required|string|max:255',
            'site_tagline' => 'nullable|string|max:255',
            'contact_email' => 'nullable|email|max:255',
            'pagination_count' => 'required|integer|min:1',
            'auto_approve_companies' => 'nullable|boolean',
            'auto_approve_reviews' => 'nullable|boolean', // Added
            'currency_symbol' => 'nullable|string|max:5',
            'currency_code' => 'nullable|string|max:3',

            'social_facebook_url' => 'nullable|url|max:255',
            'social_twitter_url' => 'nullable|url|max:255',
            'social_linkedin_url' => 'nullable|url|max:255',
            'social_instagram_url' => 'nullable|url|max:255',

            'homepage_hero_title' => 'nullable|string|max:255',
            'homepage_hero_subtitle' => 'nullable|string|max:500',
            'homepage_hero_search_placeholder' => 'nullable|string|max:100',

            'hero_slide_1_title' => 'nullable|string|max:255',
            'hero_slide_1_subtitle' => 'nullable|string|max:500',
            'hero_slide_1_button_text' => 'nullable|string|max:100',
            'hero_slide_1_button_link' => 'nullable|string|max:255',
            'hero_slide_1_image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',

            'hero_slide_2_title' => 'nullable|string|max:255',
            'hero_slide_2_subtitle' => 'nullable|string|max:500',
            'hero_slide_2_button_text' => 'nullable|string|max:100',
            'hero_slide_2_button_link' => 'nullable|string|max:255',
            'hero_slide_2_image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',

            'homepage_featured_companies_count' => 'nullable|integer|min:0',
            'homepage_featured_products_count' => 'nullable|integer|min:0',
            'homepage_latest_tenders_count' => 'nullable|integer|min:0',
            'homepage_latest_jobs_count' => 'nullable|integer|min:0',
        ]);

        // Update all validated string/numeric settings directly
        foreach ($validatedData as $key => $value) {
            if ($request->hasFile($key)) { // Skip file fields for now
                continue;
            }
            // For boolean fields not explicitly validated as boolean but present in form (checkboxes)
            if (array_key_exists($key, $settings->getAttributes()) && is_bool($settings->getAttributes()[$key]) && !$request->has($key)) {
                $settings->$key = false;
            } elseif (is_string($value) || is_numeric($value) || is_null($value) || is_bool($value)) {
                 $settings->$key = $value;
            }
        }

        $settings->auto_approve_companies = $request->boolean('auto_approve_companies');
        $settings->auto_approve_reviews = $request->boolean('auto_approve_reviews'); // Added

        // Handle file uploads for slider images
        foreach (['hero_slide_1_image', 'hero_slide_2_image'] as $imageField) {
            if ($request->hasFile($imageField)) {
                if ($settings->$imageField && Storage::disk('public')->exists($settings->$imageField)) {
                    Storage::disk('public')->delete($settings->$imageField);
                }
                $path = $request->file($imageField)->store('settings/hero_slider', 'public');
                $settings->$imageField = $path;
            } elseif ($request->has("remove_{$imageField}")) {
                if ($settings->$imageField && Storage::disk('public')->exists($settings->$imageField)) {
                    Storage::disk('public')->delete($settings->$imageField);
                }
                $settings->$imageField = null;
            }
        }

        $settings->save();

        return redirect()->route('admin.businessdirectory.settings.edit')
                         ->with('success', 'Settings updated successfully.');
    }
}