<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\User; // Standardized User model path

class Order extends Model
{
    use HasFactory;

    protected $table = 'bd_orders';

    protected $fillable = [
        'user_id', // Buyer
        'company_id', // Seller
        'order_number',
        'total_amount',
        'currency',
        'status', // e.g., pending, processing, shipped, delivered, cancelled, refunded
        'payment_method',
        'payment_status', // e.g., pending, paid, failed
        'shipping_address', // JSON
        'billing_address', // JSON
        'notes',
    ];

    protected $casts = [
        'total_amount' => 'decimal:2',
        'shipping_address' => 'array',
        'billing_address' => 'array',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            if (empty($order->order_number)) {
                // Generate a unique order number, e.g., ORD-YYYYMMDD-XXXXX
                $order->order_number = 'ORD-' . now()->format('Ymd') . '-' . strtoupper(uniqid());
            }
        });
    }

    /**
     * The user (buyer) who placed the order.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * The company (seller) fulfilling the order.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    /**
     * The items included in the order.
     */
    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Recalculate the total amount based on order items.
     * Useful if items are modified after order creation (though typically they shouldn't be).
     */
    public function recalculateTotal()
    {
        $this->total_amount = $this->items->sum('total_price');
        $this->save();
    }

    protected static function newFactory() {
        return \Modules\BusinessDirectory\Database\factories\OrderFactory::new();
    }

    /**
     * Get the defined statuses for an order.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'pending' => 'Pending',
            'processing' => 'Processing',
            'shipped' => 'Shipped',
            'delivered' => 'Delivered',
            'completed' => 'Completed',
            'cancelled' => 'Cancelled',
            'refunded' => 'Refunded',
            'payment_failed' => 'Payment Failed',
        ];
    }

    /**
     * Get the defined payment statuses for an order.
     *
     * @return array
     */
    public static function getPaymentStatuses(): array
    {
        return [
            'pending' => 'Pending',
            'paid' => 'Paid',
            'failed' => 'Failed',
            'refunded' => 'Refunded',
        ];
    }
}
