<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\User; // Standardized User model path

class JobApplication extends Model
{
    use HasFactory;

    protected $table = 'bd_job_applications';

    protected $fillable = [
        'job_id',
        'user_id',
        'resume_path',
        'cover_letter',
        'status', // e.g., applied, shortlisted, interviewed, rejected, hired
        'applied_at',
    ];

    protected $casts = [
        'applied_at' => 'datetime',
    ];

    /**
     * The job this application is for.
     */
    public function job(): BelongsTo
    {
        return $this->belongsTo(Job::class);
    }

    /**
     * The user who submitted the application.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Set the applied_at attribute.
     * If no value is provided, it defaults to the current time.
     *
     * @param  mixed  $value
     * @return void
     */
    public function setAppliedAtAttribute($value)
    {
        $this->attributes['applied_at'] = $value ?? now();
    }

    /**
     * Factory
     */
    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\JobApplicationFactory::new();
    }

    /**
     * Get the defined statuses for a job application.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'applied' => 'Applied',
            'shortlisted' => 'Shortlisted',
            'interviewed' => 'Interviewed',
            'rejected' => 'Rejected',
            'hired' => 'Hired',
            'withdrawn' => 'Withdrawn', // Applicant withdrew
        ];
    }
}
