<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\User;

class Job extends Model
{
    use HasFactory, HasSlug;

    protected $table = 'bd_jobs';

    protected $fillable = [
        'company_id',
        'category_id',
        'user_id', // User who posted the job
        'title',
        'slug',
        'description',
        'job_type', // e.g., full-time, part-time, contract
        'location',
        'salary_min',
        'salary_max',
        'currency',
        'experience_level',
        'education_level',
        'skills_required',
        'application_deadline',
        'status', // e.g., open, closed, filled
        'is_remote',
        'views_count',
    ];

    protected $casts = [
        'application_deadline' => 'date',
        'is_remote' => 'boolean',
        'views_count' => 'integer',
        'salary_min' => 'decimal:2',
        'salary_max' => 'decimal:2',
        'skills_required' => 'array',
    ];



    /**
     * Get the options for generating the slug.
     */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('title')
            ->saveSlugsTo('slug');
    }

        // Define status constants
    public const STATUS_DRAFT = 'draft';
    public const STATUS_OPEN = 'open';
    public const STATUS_CLOSED = 'closed';
    public const STATUS_FILLED = 'filled';
    public const STATUS_EXPIRED = 'expired';

    /**
     * Get the company that posted the job.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the user who posted the job.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the applications for the job.
     */
    public function jobApplications(): HasMany
    {
        return $this->hasMany(JobApplication::class);
    }

    /**
     * Get all of the job's attachments.
     */
    public function attachments(): MorphMany
    {
        return $this->morphMany(Attachment::class, 'attachable');
    }
    
    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\JobFactory::new();
    }

    /**
     * Get the defined statuses for a job.
     */
    public static function getStatuses(): array
    {
        return [
            'draft' => 'Draft',
            'open' => 'Open',
            'closed' => 'Closed',
            'filled' => 'Filled',
            // 'pending_approval' => 'Pending Approval', // If admin approval is needed
        ];
    }

    /**
     * Get the defined job types.
     */
    public static function getJobTypes(): array
    {
        return [
            'full-time' => 'Full-time',
            'part-time' => 'Part-time',
            'contract' => 'Contract',
            'internship' => 'Internship',
            'temporary' => 'Temporary',
            'freelance' => 'Freelance',
            'volunteer' => 'Volunteer',
        ];
    }

    /**
     * Scope a query to only include open jobs.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOpen($query)
    {
        return $query->where('status', self::STATUS_OPEN)
                     ->where(fn($q) => $q->whereNull('application_deadline')->orWhere('application_deadline', '>=', now()));
    }
}
