<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\User; // Corrected to standard User model location
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Storage;

class Company extends Model
{
    use HasFactory, SoftDeletes, HasSlug;
    
    // Define status constants
    public const STATUS_APPROVED = 'approved';
    public const STATUS_PENDING_APPROVAL = 'pending_approval';
    public const STATUS_REJECTED = 'rejected';
    public const STATUS_SUSPENDED = 'suspended';
    // Add any other statuses you might use, e.g., 'draft'

    protected $table = 'bd_companies';

    protected $fillable = [
        'user_id', // Owner/Creator of the company profile
        'company_type_id',
        'name', // Required
        'slug', 
        'logo_path', // Path to logo
        'banner_image_path', // Path to banner
        'description', // Required
        'address_line_1', // Recommended: Use specific address fields
        'address_line_2', // Optional
        'city', // Recommended
        'state_province', // Recommended
        'postal_code', // Recommended
        'country_code', // Recommended (e.g., US, GB)
        'latitude',
        'longitude',
        'phone_number', // Primary phone
        'email', // Required
        'website_url', // Full website URL
        'whatsapp_number',
        'established_date',
        'employee_count',
        'annual_revenue',
        'social_links', // JSON
        'meta_title',
        'meta_description',
        'meta_keywords',
        'status', // e.g., pending, approved, rejected, suspended
        'category_id', // Added
        'is_featured',
        'is_verified', // Added
        'is_ecommerce', // Added
        'views_count',
        'approved_at',
        'rejected_at',
        'rejection_reason',
        'suspended_at',
        'suspension_reason',
    ];

    protected $casts = [
        'established_date' => 'date',
        'social_links' => 'array',
        'is_featured' => 'boolean',
        'is_verified' => 'boolean',
        'views_count' => 'integer',
        'annual_revenue' => 'decimal:2',
        'latitude' => 'decimal:7', // Matched table definition (10,7)
        'longitude' => 'decimal:7',// Matched table definition (10,7)
        'is_ecommerce' => 'boolean', // Added
    ];

    protected $appends = [
        'logo_url_public', 'banner_url_public'
    ];

    protected $dates = [ // For Carbon instance casting, SoftDeletes handles deleted_at
        'approved_at', 'rejected_at', 'suspended_at'
    ];


    /**
     * Get the options for generating the slug.
     */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('name')
            ->saveSlugsTo('slug');
    }

    /**
     * Get the user who owns/created this company profile.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the type of the company.
     */
    public function companyType(): BelongsTo
    {
        return $this->belongsTo(CompanyType::class);
    }

    /**
     * Get the category of the company.
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Get the products listed by this company.
     */
    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Get the tenders posted by this company.
     */
    public function tenders(): HasMany
    {
        return $this->hasMany(Tender::class);
    }

    /**
     * Get the jobs posted by this company.
     */
    public function jobs(): HasMany
    {
        return $this->hasMany(Job::class);
    }

    /**
     * Get the projects managed by this company.
     */
    public function projects(): HasMany
    {
        return $this->hasMany(CompanyProject::class, 'company_id');
    }

    /**
     * Get the reviews for this company.
     */
    public function reviews(): HasMany
    {
        return $this->hasMany(Review::class);
    }

    /**
     * Get the orders where this company is the seller.
     */
    public function ordersReceived(): HasMany
    {
        return $this->hasMany(Order::class, 'company_id');
    }

    /**
     * Accessor for approved reviews.
     */
    public function getApprovedReviewsAttribute()
    {
        return $this->reviews()->where('is_approved', true)->get();
    }

    /**
     * Accessor for public logo URL.
     */
    public function getLogoUrlPublicAttribute(): string
    {
        if ($this->logo_path && Storage::disk('public')->exists($this->logo_path)) {
            return Storage::url($this->logo_path);
        }
        return asset(config('businessdirectory.default_logo_path', 'images/defaults/company_logo.png'));
    }

    /**
     * Accessor for public banner URL.
     */
    public function getBannerUrlPublicAttribute(): string
    {
        if ($this->banner_image_path && Storage::disk('public')->exists($this->banner_image_path)) {
            return Storage::url($this->banner_image_path);
        }
        return asset(config('businessdirectory.default_banner_path', 'images/defaults/company_banner.png'));
    }

    /**
     * Check if the company has a custom logo.
     */
    public function hasCustomLogo(): bool
    {
        return !empty($this->logo_path) && Storage::disk('public')->exists($this->logo_path);
    }

    /**
     * Check if the company has a custom banner.
     */
    public function hasCustomBanner(): bool
    {
        return !empty($this->banner_image_path) && Storage::disk('public')->exists($this->banner_image_path);
    }

    // Old accessors for 'logo' and 'banner' columns (if they are still used or for backward compatibility during transition)
    // If 'logo' and 'banner' are fully deprecated and data migrated to 'logo_path' and 'banner_image_path', these can be removed.
    public function getLogoAttribute($value) {
        return $this->logo_path ? Storage::url($this->logo_path) : asset(config('businessdirectory.default_logo_path', 'images/defaults/company_logo.png'));
    }

    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\CompanyFactory::new();
    }

    /**
     * Get the defined statuses for a company.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'pending' => 'Pending Approval',
            'approved' => 'Approved',
            'rejected' => 'Rejected',
            'suspended' => 'Suspended',
            // Add other statuses if needed, e.g., 'draft', 'expired'
        ];
    }

    /**
     * Scope a query to only include approved companies.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    /**
     * Scope a query to only include featured companies.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }
}
