<?php

namespace Modules\BusinessDirectory\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\User; // Standardized User model path

class Bid extends Model
{
    use HasFactory;

    protected $table = 'bd_bids';

    protected $fillable = [
        'tender_id',
        'bidder_company_id', // Bidding company (nullable if an individual user can bid)
        'bidder_user_id',    // User who submitted the bid
        'proposal_details',  // The actual proposal text
        'bid_amount',        // The monetary amount of the bid
        'bid_currency',      // Currency for the bid_amount
        'delivery_timeframe',
        'attachments', // JSON
        'status',
        'is_flagged',
    ];

    protected $casts = [
        'bid_amount' => 'decimal:2',
        'attachments' => 'array',
        'is_flagged' => 'boolean',
    ];

    /**
     * Get the tender this bid is for.
     */
    public function tender(): BelongsTo
    {
        return $this->belongsTo(Tender::class);
    }

    /**
     * Get the company that submitted the bid (if applicable).
     */
    public function bidderCompany(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'bidder_company_id')->withDefault();
    }

    /**
     * Get the user who submitted the bid.
     */
    public function bidderUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'bidder_user_id');
    }

    protected static function newFactory()
    {
        return \Modules\BusinessDirectory\Database\factories\BidFactory::new();
    }

    /**
     * Get the defined statuses for a bid.
     *
     * @return array
     */
    public static function getStatuses(): array
    {
        return [
            'submitted' => 'Submitted',
            'viewed' => 'Viewed',
            'shortlisted' => 'Shortlisted',
            'accepted' => 'Accepted',
            'rejected' => 'Rejected',
            'withdrawn' => 'Withdrawn',
        ];
    }
}
