<?php

use App\Http\Middleware\IsAdminMiddleware; // Import the general admin middleware
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\ModuleController;
use App\Http\Controllers\Admin\SettingController; // Singular, matching corrected class
use App\Http\Controllers\Admin\RoleController; // Add this line
use App\Http\Controllers\Admin\TicketController as AdminTicketController;
use App\Http\Controllers\Admin\NotificationController as AdminNotificationController;
use App\Http\Controllers\Admin\AdminUserController; // Correct controller name
use App\Http\Controllers\Admin\SubscriptionPlanController;
use App\Http\Controllers\Admin\FeatureCreditCostController; // Added
use App\Http\Middleware\EnsureUserIsSuperAdminForManagement; // Import the middleware class
use App\Http\Controllers\Admin\AdminCreditTransactionController; // Added
use App\Http\Controllers\Admin\AdminWalletTransactionController; // Added
use App\Http\Controllers\Admin\AdminWalletWithdrawalRequestController; // Added
use App\Http\Controllers\Admin\NewsletterController as AdminNewsletterController; // Added
use App\Http\Controllers\Admin\NewsletterSubscriptionController; // Added


/*
|--------------------------------------------------------------------------
| Admin Routes
|--------------------------------------------------------------------------
|
| Here is where you can register admin routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| is assigned the "web" and "auth.admin" (or direct class path) middleware group.
|
*/

// Group all admin routes under IsAdminMiddleware
Route::middleware([IsAdminMiddleware::class])->group(function () {
    // Admin Dashboard
    Route::get('/', [DashboardController::class, 'index'])->name('dashboard');

    // Payment Gateways Dashboard
    Route::get('/payment-gateways-dashboard', [DashboardController::class, 'paymentGatewaysDashboard'])
         ->name('payment-gateways.dashboard');

    // Modules Management
    Route::get('/modules', [ModuleController::class, 'index'])->name('modules.index');
    Route::get('/modules/install', [ModuleController::class, 'showInstallForm'])->name('modules.install.form');
    Route::post('/modules/upload', [ModuleController::class, 'uploadModule'])->name('modules.upload');
    Route::post('/modules/{moduleName}/enable', [ModuleController::class, 'enableModule'])->name('modules.enable');
    Route::post('/modules/{moduleName}/disable', [ModuleController::class, 'disableModule'])->name('modules.disable');
    Route::delete('/modules/{moduleName}/uninstall', [ModuleController::class, 'uninstallModule'])->name('modules.uninstall');

    // Settings Management
    Route::get('/settings', [SettingController::class, 'index'])->name('settings.index');
    Route::put('/settings', [SettingController::class, 'update'])->name('settings.update');
    Route::post('/settings/test-mail', [SettingController::class, 'testMail'])->name('settings.testMail');

    // Ticket Management
    Route::resource('tickets', AdminTicketController::class)->except(['create', 'store']); // Admins don't create tickets this way
    Route::post('tickets/{ticket}/reply', [AdminTicketController::class, 'reply'])->name('tickets.reply');
    Route::post('tickets/{ticket}/assign', [AdminTicketController::class, 'assign'])->name('tickets.assign');
    Route::post('tickets/{ticket}/status', [AdminTicketController::class, 'updateStatus'])->name('tickets.status');

    // Notification Management
    Route::get('/notifications', [AdminNotificationController::class, 'index'])->name('notifications.index');
    Route::post('/notifications/{notificationId}/mark-as-read', [AdminNotificationController::class, 'markAsRead'])->name('notifications.markAsRead');
    Route::post('/notifications/mark-all-as-read', [AdminNotificationController::class, 'markAllAsRead'])->name('notifications.markAllAsRead');

    // User Management
    // Routes requiring Super Admin privileges for user management actions
    Route::group(['middleware' => [EnsureUserIsSuperAdminForManagement::class]], function () {
        // Resource routes except 'index' (create, store, show, edit, update, destroy)
        Route::resource('users', AdminUserController::class)->except(['index']);
        // Specific actions requiring Super Admin
        Route::post('users/{user}/suspend', [AdminUserController::class, 'suspend'])->name('users.suspend');
        Route::post('users/{user}/unsuspend', [AdminUserController::class, 'unsuspend'])->name('users.unsuspend');
        // User Subscription Management by Super Admin
        Route::post('users/{user}/subscription', [AdminUserController::class, 'updateUserSubscription'])->name('users.subscription.update');
        Route::post('users/{user}/subscription/remove', [AdminUserController::class, 'removeUserSubscription'])->name('users.subscription.remove');
        // User Credit & Wallet Balance Management by Super Admin
        Route::post('users/{user}/credits/adjust', [AdminUserController::class, 'adjustUserCredits'])->name('users.credits.adjust');
        Route::post('users/{user}/wallet/adjust', [AdminUserController::class, 'adjustUserWallet'])->name('users.wallet.adjust');


        // Magic Login (Login as User)
        Route::get('users/{user}/login-as', [AdminUserController::class, 'loginAsUser'])->name('users.login-as');
    });
    // The index route for users is accessible to regular admins (covered by the outer IsAdminMiddleware group)
    Route::get('users', [AdminUserController::class, 'index'])->name('users.index');
    // Route for admin to return to their session (needs to be accessible when impersonating)
    // This route should ideally not be under the admin prefix if the admin is logged out during impersonation.
    // Let's place it outside the admin group but still protect it.

    // Feature Credit Costs Management (Protected by Super Admin middleware in controller constructor)
    Route::get('/feature-credit-costs', [FeatureCreditCostController::class, 'index'])->name('feature-credit-costs.index');
    Route::get('/feature-credit-costs/create', [FeatureCreditCostController::class, 'create'])->name('feature-credit-costs.create');
    Route::post('/feature-credit-costs', [FeatureCreditCostController::class, 'store'])->name('feature-credit-costs.store');
    Route::get('/feature-credit-costs/{featureCreditCost}/edit', [FeatureCreditCostController::class, 'edit'])->name('feature-credit-costs.edit');
    // Admin Transaction Logs (Super Admin only - enforced in controller)
    Route::get('/credit-transactions', [AdminCreditTransactionController::class, 'index'])->name('credit-transactions.index');
    Route::get('/wallet-transactions', [AdminWalletTransactionController::class, 'index'])->name('wallet-transactions.index');
    // Wallet Withdrawal Requests Management (Super Admin only - enforced in controller)
    Route::get('wallet-withdrawal-requests', [AdminWalletWithdrawalRequestController::class, 'index'])->name('wallet-withdrawal-requests.index');
    Route::get('wallet-withdrawal-requests/{withdrawalRequest}', [AdminWalletWithdrawalRequestController::class, 'show'])->name('wallet-withdrawal-requests.show');
    Route::post('wallet-withdrawal-requests/{withdrawalRequest}/update-status', [AdminWalletWithdrawalRequestController::class, 'updateStatus'])->name('wallet-withdrawal-requests.updateStatus');


    Route::put('/feature-credit-costs/{featureCreditCost}', [FeatureCreditCostController::class, 'update'])->name('feature-credit-costs.update');
    Route::delete('/feature-credit-costs/{featureCreditCost}', [FeatureCreditCostController::class, 'destroy'])->name('feature-credit-costs.destroy');

    // Subscription Plan Management
    Route::resource('subscription-plans', SubscriptionPlanController::class); // Removed ->names() as it's usually not needed for simple resource

    // Role Management (ensure this is protected, e.g., by Super Admin middleware if not all admins should access)
    Route::resource('roles', RoleController::class);

    // Newsletter Management (Composing & Sending)
    Route::resource('newsletters', AdminNewsletterController::class);
    Route::post('newsletters/{newsletter}/send-test', [AdminNewsletterController::class, 'sendTest'])->name('newsletters.sendTest');
    Route::post('newsletters/{newsletter}/dispatch-send', [AdminNewsletterController::class, 'dispatchSend'])->name('newsletters.dispatchSend');

    // Newsletter Subscriptions Management
    Route::resource('newsletter-subscriptions', NewsletterSubscriptionController::class);
});

// This route is outside the admin prefix and IsAdminMiddleware group
// because the admin is technically logged out when impersonating.
// It relies on session data for authentication back to admin.
Route::get('/return-to-admin', [AdminUserController::class, 'returnToAdmin'])->name('admin.return-to-admin')->middleware('web');

// Add other admin routes here
