{{--
    Variables expected:
    $groupDisplayName (string) - The display name of the main group (e.g., "General", "Mail Settings")
    $originalGroupName (string) - The original key of the main group
    $settingsGroupedBySubGroup (Collection) - Collection of [SubGroupName => CollectionOfSettings]
    $groupDescription (string|null) - Description for the main group
    $canEditGroup (bool) - Whether the current user can edit settings in this main group
--}}

{{-- This whole block is for ONE main group (e.g., "General Settings" sub-tab) --}}

{{-- Display Main Group Title and Description --}}
<div> {{-- This div will be a child of the space-y-6 parent from index.blade.php --}}
    <h2 class="text-2xl font-semibold text-gray-800 dark:text-gray-200">{{ $groupDisplayName }}</h2>
    @if($groupDescription)
        <p class="mt-1 text-sm text-gray-600 dark:text-gray-400">{{ $groupDescription }}</p>
    @endif
</div>

{{-- Separator if there are sub-groups to follow --}}
@if($settingsGroupedBySubGroup->count() > 0)
    <hr class="border-gray-200 dark:border-gray-700"> {{-- This hr will also get space-y-6 margin from parent --}}
@endif

{{-- Loop through Sub-Groups. Each sub-group is its own card. --}}
@forelse($settingsGroupedBySubGroup as $subGroupName => $settingsInSubGroup)
    <div class="bg-white dark:bg-gray-800 shadow-xl rounded-xl overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
            <h4 class="text-md font-semibold text-gray-800 dark:text-gray-200">
                {{ $subGroupName }}
            </h4>
            {{-- Optional: Sub-group description can be stored on the Setting model if needed, or derived --}}
        </div>
        <div class="px-6 py-6">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-8">
                @foreach ($settingsInSubGroup as $setting)
                    @php $settingKeyName = 'settings_values[' . $setting->key . ']'; @endphp
                    <input type="hidden" name="settings_keys[]" value="{{ $setting->key }}">

                    <div class="{{ $setting->type === 'textarea' || $setting->type === 'checkbox' ? 'md:col-span-2' : 'md:col-span-1' }}">
                        @if ($setting->type !== 'checkbox')
                            <label for="{{ $settingKeyName }}" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">{{ $setting->name }}</label>
                        @endif

                        @if ($setting->type === 'text' || $setting->type === 'number' || $setting->type === 'email' || $setting->type === 'password')
                            <input type="{{ $setting->type === 'password' ? 'password' : ($setting->type === 'number' ? 'number' : 'text') }}" name="{{ $settingKeyName }}" id="{{ $settingKeyName }}" value="{{ old($settingKeyName, $setting->type === 'password' ? '' : $setting->value) }}"
                                   class="block w-full shadow-sm sm:text-sm border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-gray-200 focus:ring-blue-500 focus:border-blue-500 disabled:opacity-60 disabled:cursor-not-allowed"
                                   placeholder="{{ $setting->type === 'password' ? 'Enter new password to change' : ($setting->placeholder ?? '') }}" {{ !$canEditGroup ? 'disabled' : '' }}>
                        @elseif ($setting->type === 'textarea')
                            <textarea name="{{ $settingKeyName }}" id="{{ $settingKeyName }}" rows="4"
                                      class="block w-full shadow-sm sm:text-sm border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-gray-200 focus:ring-blue-500 focus:border-blue-500 disabled:opacity-60 disabled:cursor-not-allowed" placeholder="{{ $setting->placeholder ?? '' }}" {{ !$canEditGroup ? 'disabled' : '' }}>{{ old($settingKeyName, $setting->value) }}</textarea>
                        @elseif ($setting->type === 'checkbox')
                            <label for="{{ $settingKeyName }}_checkbox" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">{{ $setting->name }}</label>
                            <div class="flex items-center mt-1">
                                <input type="hidden" name="{{ $settingKeyName }}" value="0" {{ !$canEditGroup ? 'disabled' : '' }}>
                                <input id="{{ $settingKeyName }}_checkbox" name="{{ $settingKeyName }}" type="checkbox" value="1" {{ old($settingKeyName, $setting->value) == '1' ? 'checked' : '' }}
                                       class="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 dark:border-gray-600 rounded dark:bg-gray-700 disabled:opacity-60 disabled:cursor-not-allowed" {{ !$canEditGroup ? 'disabled' : '' }}>
                                <label for="{{ $settingKeyName }}_checkbox" class="ml-2 text-sm text-gray-700 dark:text-gray-300">{{ $setting->placeholder ?: 'Enable' }}</label>
                            </div>
                        @elseif ($setting->type === 'file')
                            <input type="file" name="settings_files[{{ $setting->key }}]" id="settings_files_{{ $setting->key }}"
                                   class="block w-full text-sm text-gray-900 dark:text-gray-100 border border-gray-300 dark:border-gray-600 rounded-md cursor-pointer bg-gray-50 dark:bg-gray-700 focus:outline-none focus:border-blue-500 focus:ring-blue-500 file:mr-4 file:py-2 file:px-4 file:rounded-md file:border-0 file:text-sm file:font-semibold file:bg-blue-50 dark:file:bg-blue-700 file:text-blue-700 dark:file:text-blue-100 hover:file:bg-blue-100 dark:hover:file:bg-blue-600 disabled:opacity-60 disabled:cursor-not-allowed" {{ !$canEditGroup ? 'disabled' : '' }}>
                            @if ($setting->value && Storage::disk('public')->exists($setting->value))
                                <div class="mt-2 text-sm">
                                    <span class="text-gray-500 dark:text-gray-400">Current:</span>
                                    @if(in_array(strtolower(pathinfo($setting->value, PATHINFO_EXTENSION)), ['png', 'jpg', 'jpeg', 'gif', 'svg', 'ico', 'webp']))
                                        <img src="{{ Storage::url($setting->value) }}" alt="{{ $setting->name }}" class="h-16 mt-1 max-w-xs object-contain rounded-md border dark:border-gray-600">
                                    @else
                                        <a href="{{ Storage::url($setting->value) }}" target="_blank" class="ml-1 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 underline">{{ basename($setting->value) }}</a>
                                    @endif
                                </div>
                            @endif
                        @elseif ($setting->type === 'select' && isset($setting->options) && is_array($setting->options))
                            <select name="{{ $settingKeyName }}" id="{{ $settingKeyName }}" class="block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md dark:bg-gray-700 dark:text-gray-200 disabled:opacity-60 disabled:cursor-not-allowed" {{ !$canEditGroup ? 'disabled' : '' }}>
                                @if($setting->placeholder)
                                    <option value="">{{ $setting->placeholder }}</option>
                                @endif
                                @foreach($setting->options as $optValue => $optLabel)
                                    <option value="{{ $optValue }}" {{ old($settingKeyName, $setting->value) == $optValue ? 'selected' : '' }}>{{ $optLabel }}</option>
                                @endforeach
                            </select>
                        @else {{-- Fallback for unknown types or simple text --}}
                            <input type="text" name="{{ $settingKeyName }}" id="{{ $settingKeyName }}" value="{{ old($settingKeyName, $setting->value) }}"
                                   class="block w-full shadow-sm sm:text-sm border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-gray-200 focus:ring-blue-500 focus:border-blue-500 disabled:opacity-60 disabled:cursor-not-allowed" placeholder="{{ $setting->placeholder ?? '' }}" {{ !$canEditGroup ? 'disabled' : '' }}>
                        @endif
                        <div class="mt-1">
                            @error($settingKeyName) <p class="text-xs text-red-600 dark:text-red-400">{{ $message }}</p> @enderror
                            @error('settings_files.' . $setting->key) <p class="text-xs text-red-600 dark:text-red-400">{{ $message }}</p> @enderror
                        </div>
                        @if($setting->description)
                            <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ $setting->description }}</p>
                        @endif
                    </div>
                @endforeach
            </div>
        </div>
    </div>
@empty
    {{-- This block is rendered if $settingsGroupedBySubGroup is empty. --}}
    {{-- The main group title/description div above would have already been rendered. --}}
    <div class="bg-white dark:bg-gray-800 shadow-xl rounded-xl overflow-hidden p-6">
        <p class="text-center text-gray-500 dark:text-gray-400">No settings are available in the '{{ $groupDisplayName }}' group.</p>
    </div>
@endforelse
