<?php

namespace App\Services;

use App\Models\User;
use App\Models\WalletTransaction;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Brick\Math\BigDecimal;
use Brick\Math\RoundingMode;

class WalletService
{
    public function deposit(
        User $user,
        float $amount,
        string $currency,
        string $gateway,
        ?string $gatewayTxId,
        string $description,
        string $status = 'completed',
        ?Model $relatedObject = null,
        ?array $metadata = null
    ): WalletTransaction {
        if ($amount <= 0) {
            throw new \InvalidArgumentException('Deposit amount must be positive.');
        }

        return DB::transaction(function () use ($user, $amount, $currency, $gateway, $gatewayTxId, $description, $status, $relatedObject, $metadata) {
            // Ensure $user->wallet_balance is always a string for BigDecimal
            $currentBalance = BigDecimal::of((string) ($user->wallet_balance ?? '0'));
            $depositAmount = BigDecimal::of((string) $amount); // Ensure $amount is treated as string for BigDecimal
            $newBalance = $currentBalance->plus($depositAmount);

            if ($status === 'completed') {
                $user->wallet_balance = $newBalance->toScale(2, RoundingMode::HALF_UP);
                $user->save();
            }

            return $this->logTransaction(
                $user,
                'deposit',
                $amount,
                $newBalance->toFloat(),
                $currency,
                $description,
                $gateway,
                $gatewayTxId,
                $status,
                $relatedObject,
                $metadata
            );
        });
    }

    public function spendFromWallet(User $user, float $amount, string $type, string $description, ?Model $relatedObject = null, string $currency = 'USD'): ?WalletTransaction
    {
        if ($amount <= 0) {
            throw new \InvalidArgumentException('Spend amount must be positive.');
        }

        // Ensure $user->wallet_balance is always a string for BigDecimal
        $userBalance = BigDecimal::of((string) ($user->wallet_balance ?? '0'));
        $spendAmount = BigDecimal::of((string) $amount);

        if ($userBalance->isLessThan($spendAmount)) {
            Log::warning("User {$user->id} has insufficient wallet balance to spend {$amount} {$currency}. Balance: {$user->wallet_balance}");
            return null;
        }

        return DB::transaction(function () use ($user, $amount, $type, $description, $relatedObject, $currency, $spendAmount) {
            // Re-fetch inside transaction for safety, also null-coalesce
            $currentBalance = BigDecimal::of((string) ($user->wallet_balance ?? '0'));
            $newBalance = $currentBalance->minus($spendAmount);

            $user->wallet_balance = $newBalance->toScale(2, RoundingMode::HALF_UP);
            $user->save();

            return $this->logTransaction(
                $user,
                $type,
                -$amount,
                $newBalance->toFloat(),
                $currency,
                $description,
                'wallet',
                null,
                'completed',
                $relatedObject
            );
        });
    }

    public function adjustBalance(User $user, float $amount, string $description, string $currency = 'USD', ?Model $relatedObject = null): WalletTransaction
    {
        // The request form allows "0.03" which is valid.
        // Your controller casting to (float) makes it 0.03.
        // If the original issue was that amount was null here due to controller issue,
        // the float type hint should have caught it higher up.
        // So the 'null' must be coming from $user->wallet_balance.

        if ($amount == 0) {
            throw new \InvalidArgumentException('Adjustment amount cannot be zero.');
        }

        return DB::transaction(function () use ($user, $amount, $description, $currency, $relatedObject) {
            // LINE 137 (was): $currentBalance = BigDecimal::of($user->wallet_balance);
            // Corrected to handle potential null $user->wallet_balance
            $currentBalance = BigDecimal::of((string) ($user->wallet_balance ?? '0'));

            // LINE 138 (was): $adjustmentAmount = BigDecimal::of($amount);
            // Corrected to ensure $amount is string for BigDecimal, though float type hint should handle this.
            $adjustmentAmount = BigDecimal::of((string) $amount);

            $newBalance = $currentBalance->plus($adjustmentAmount);

            // This check is important, ensure admin adjustments don't unintentionally go negative
            if ($newBalance->isLessThan(BigDecimal::zero()) && $amount < 0) { // Only throw if deducting below zero
                Log::error("Attempted admin adjustment for user {$user->id} would result in negative balance: Current {$currentBalance}, Adjustment {$amount}.");
                throw new \RuntimeException("Admin adjustment would result in a negative wallet balance for user {$user->id}.");
            }

            $user->wallet_balance = $newBalance->toScale(2, RoundingMode::HALF_UP);
            $user->save();

            $transactionType = $amount > 0 ? 'admin_adjustment_credit' : 'admin_adjustment_debit';

            return $this->logTransaction(
                $user, $transactionType, $amount, $newBalance->toFloat(), $currency,
                $description, 'admin', null, 'completed', $relatedObject
            );
        });
    }

    public function getUserWalletBalance(User $user, string $currency = 'USD'): float
    {
        // Assuming single currency wallet for now. Multi-currency would require more complex logic.
        return (float) ($user->wallet_balance ?? 0); // Ensure it's not null before casting to float
    }

    private function logTransaction(User $user, string $type, float $amount, float $balanceAfter, string $currency, string $description, ?string $gateway, ?string $gatewayTxId, string $status = 'completed', ?Model $relatedObject = null, ?array $metadata = null): WalletTransaction
    {
        $transactionData = [
            'user_id' => $user->id,
            'type' => $type,
            'amount' => $amount,
            'balance_after_transaction' => $balanceAfter,
            'currency' => $currency,
            'payment_gateway' => $gateway,
            'gateway_transaction_id' => $gatewayTxId,
            'description' => $description,
            'status' => $status,
            'metadata' => $metadata,
        ];

        if ($relatedObject) {
            $transactionData['related_id'] = $relatedObject->getKey();
            $transactionData['related_type'] = $relatedObject->getMorphClass();
        }

        return WalletTransaction::create($transactionData);
    }
}