<?php

namespace App\Services;

use App\Models\User;
use App\Models\UserCreditTransaction;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CreditService
{
    /**
     * Award credits to a user.
     *
     * @param User $user The user to award credits to.
     * @param int $amount The amount of credits to award (positive integer).
     * @param string $type The type of transaction (e.g., 'award_subscription', 'award_purchase', 'admin_adjustment').
     * @param string $description A description for the transaction.
     * @param Model|null $relatedObject An optional related model (e.g., Subscription, Order).
     * @return UserCreditTransaction The created credit transaction.
     * @throws \Exception If awarding credits fails.
     */
    public function awardCredits(User $user, int $amount, string $type, string $description, ?Model $relatedObject = null): UserCreditTransaction
    {
        if ($amount <= 0) {
            throw new \InvalidArgumentException('Credit amount to award must be positive.');
        }

        return DB::transaction(function () use ($user, $amount, $type, $description, $relatedObject) {
            $user->increment('credit_balance', $amount);
            // It's good practice to reload the user to get the updated balance for the transaction log,
            // or calculate it manually if performance is critical for many simultaneous transactions.
            $user->refresh();

            return $this->logTransaction(
                $user,
                $type,
                $amount, // Positive for awards
                $user->credit_balance,
                $description,
                $relatedObject
            );
        });
    }

    /**
     * Spend credits from a user's balance.
     *
     * @param User $user The user spending credits.
     * @param int $amount The amount of credits to spend (positive integer).
     * @param string $type The type of transaction (e.g., 'spend_feature').
     * @param string $description A description for the transaction.
     * @param Model|null $relatedObject An optional related model (e.g., FeatureUsageLog).
     * @return UserCreditTransaction|null The created credit transaction, or null if insufficient balance.
     * @throws \Exception If spending credits fails for reasons other than insufficient balance.
     */
    public function spendCredits(User $user, int $amount, string $type, string $description, ?Model $relatedObject = null): ?UserCreditTransaction
    {
        if ($amount <= 0) {
            throw new \InvalidArgumentException('Credit amount to spend must be positive.');
        }

        if ($user->credit_balance < $amount) {
            Log::warning("User {$user->id} has insufficient credits to spend {$amount}. Balance: {$user->credit_balance}");
            return null; // Insufficient balance
        }

        return DB::transaction(function () use ($user, $amount, $type, $description, $relatedObject) {
            $user->decrement('credit_balance', $amount);
            $user->refresh();

            return $this->logTransaction(
                $user,
                $type,
                -$amount, // Negative for spends
                $user->credit_balance,
                $description,
                $relatedObject
            );
        });
    }

    /**
     * Get the user's current credit balance.
     */
    public function getUserCreditBalance(User $user): int
    {
        return (int) $user->credit_balance;
    }

    /**
     * Log a credit transaction.
     * This is a helper method, typically called within a DB transaction.
     */
    private function logTransaction(User $user, string $type, int $amount, int $balanceAfter, string $description, ?Model $relatedObject = null): UserCreditTransaction
    {
        $transactionData = [
            'user_id' => $user->id,
            'type' => $type,
            'credits_changed' => $amount, // Changed 'amount' to 'credits_changed'
            'balance_after_transaction' => $balanceAfter,
            'description' => $description,
        ];

        if ($relatedObject) {
            $transactionData['related_id'] = $relatedObject->getKey();
            $transactionData['related_type'] = $relatedObject->getMorphClass();
        }

        return UserCreditTransaction::create($transactionData);
    }
}