<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Contracts\Cache\Repository as CacheRepository;

class ActivationService
{
    protected CacheRepository $cache;

    public function __construct(CacheRepository $cache)
    {
        $this->cache = $cache;
    }

    /**
     * Checks if the boilerplate is properly activated by validating its token.
     * Caches the validation result.
     */
    public function isBoilerplateActive(): bool
    {
        // Dev mode check: If BOILERPLATE_DEV_MODE_SKIP_LICENSE_CHECK is true, always return true.
        if (config('app.boilerplate_dev_mode_skip_license_check', false)) {
            Log::debug('[ActivationService::isBoilerplateActive] Dev mode: Skipping license check.');
            return true;
        }

        $activationToken = Setting::getValue('boilerplate_activation_token');
        $activatedDomain = Setting::getValue('boilerplate_activated_domain');

        if (empty($activationToken) || empty($activatedDomain)) {
            Log::debug('[ActivationService::isBoilerplateActive] Activation token or domain not set in settings.');
            return false; // Not activated or essential info missing
        }

        $cacheKey = 'boilerplate_activation_valid_' . md5($activationToken . $activatedDomain);
        $cacheDurationHours = config('app.boilerplate_activation_cache_hours', 24);
        $cacheDuration = now()->addHours($cacheDurationHours);

        // Check cache first
        if ($this->cache->has($cacheKey)) {
            $isValid = $this->cache->get($cacheKey);
            // Log::debug("[ActivationService::isBoilerplateActive] Activation status from cache for {$activatedDomain}: " . ($isValid ? 'Valid' : 'Invalid'));
            return $isValid;
        }

        try {
            $productSlug = config('app.boilerplate_product_slug'); // Ensure this is set in config/app.php or .env
            $appManagerApiBaseUrl = rtrim(config('app.appmanager_url'), '/') . '/api/app-manager'; // Ensure this is set
            $validationApiUrl = $appManagerApiBaseUrl . '/validate-activation';

            if (empty($productSlug) || empty(config('app.appmanager_url'))) {
                Log::warning('[ActivationService::isBoilerplateActive] AppManager URL or Product Slug not configured for token validation.');
                $this->cache->put($cacheKey, false, $cacheDuration);
                return false;
            }

            $response = Http::timeout(10)->post($validationApiUrl, [
                'activation_token' => $activationToken,
                'domain' => $activatedDomain,
                'product_slug' => $productSlug,
            ]);

            if ($response->successful() && $response->json('status') === 'success' && $response->json('is_valid') === true) {
                $this->cache->put($cacheKey, true, $cacheDuration);
                return true;
            } else {
                Log::warning('[ActivationService::isBoilerplateActive] Activation token validation failed with AppManager.', ['response_body' => $response->body(), 'status_code' => $response->status()]);
                $this->cache->put($cacheKey, false, $cacheDuration);
                return false;
            }
        } catch (\Exception $e) {
            Log::error('[ActivationService::isBoilerplateActive] Error during activation token validation API call.', ['error' => $e->getMessage()]);
            // Cache as false on API error to prevent repeated calls, but maybe for a shorter duration
            $this->cache->put($cacheKey, false, now()->addMinutes(config('app.license_check_interval_minutes', 60)));
            return false;
        }
    }
}