<?php

namespace App\Providers;

use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider as ServiceProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;
use Nwidart\Modules\Facades\Module; // Import the Module facade

class RouteServiceProvider extends ServiceProvider
{
    /**
     * The path to your application's "home" route.
     *
     * Typically, users are redirected here after authentication.
     *
     * @var string
     */
    public const HOME = '/dashboard'; // This remains the default for regular users

    /**
     * Define your route model bindings, pattern filters, and other route configuration.
     */
    public function boot(): void
    {
        $this->configureRateLimiting();
        // $this->mapModuleRoutes(); // We will call this inside the routes() closure for better control
        
        $this->routes(function () {
            Route::middleware('api')
                ->prefix('api')
                ->group(base_path('routes/api.php'));

            Route::middleware('web')
                ->group(base_path('routes/web.php'));

            // Admin routes
            Route::middleware(['web', \App\Http\Middleware\IsAdminMiddleware::class]) // Use direct class path
                ->prefix('admin')
                ->as('admin.')
                ->group(base_path('routes/admin.php'));

            $this->mapModuleRoutes(); // Call the method to map module routes
        });
    }

    /**
     * Configure the rate limiters for the application.
     *
     * @return void
     */
    protected function configureRateLimiting(): void
    {
        RateLimiter::for('api', function (Request $request) {
            return Limit::perMinute(60)->by($request->user()?->id ?: $request->ip());
        });

        RateLimiter::for('login', function (Request $request) {
            return Limit::perMinute(5)->by(strtolower($request->input('email')).'|'.$request->ip());
        });

        RateLimiter::for('two-factor', function (Request $request) {
            return Limit::perMinute(5)->by($request->ip());
        });
    }

    /**
     * Define the routes for the modules.
     *
     * This method iterates through all enabled modules and registers their
     * respective RouteServiceProviders, allowing each module to define its own
     * route groups, middleware, and namespaces.
     *
     * It includes special handling for the LandingPage module to conditionally
     * load its routes based on a system setting.
     */
    protected function mapModuleRoutes(): void
    {
        if (!class_exists(Module::class)) {
            return; // Module system not available
        }

        $modulesToLoad = [];
        $landingPageModule = null;

        // First, collect all modules and identify LandingPage
        foreach (Module::getOrdered() as $module) {
            if (!$module->isEnabled()) {
                continue;
            }

            if (strtolower($module->getName()) === 'landingpage') {
                $landingPageModule = $module;
            } else {
                $modulesToLoad[] = $module;
            }
        }

        // Register RouteServiceProviders for all modules EXCEPT LandingPage initially
        foreach ($modulesToLoad as $module) {
            $this->registerModuleRouteServiceProvider($module);
        }

        // Conditionally register LandingPage's RouteServiceProvider last
        // This ensures its catch-all route has the lowest precedence if active.
        if ($landingPageModule) {
            // Check your application's setting to see if the landing page module should be active
            // The '0' as default ensures if the setting is missing, it's considered disabled.
            if (function_exists('setting') && setting('landing_page_enabled', '0') == '1') {
                $this->registerModuleRouteServiceProvider($landingPageModule);
            } else {
                // If LandingPage module is disabled by setting, but you still want
                // BusinessDirectory to handle root paths, ensure its routes are loaded.
                // This is implicitly handled if BusinessDirectory is in $modulesToLoad.
                // If BusinessDirectory was also skipped above for some reason, you might need
                // to explicitly load its RouteServiceProvider here.
            }
        }
    }

    /**
     * Helper to register a module's RouteServiceProvider.
     */
    protected function registerModuleRouteServiceProvider(\Nwidart\Modules\Module $module): void
    {
        // Construct the fully qualified namespace for the module's RouteServiceProvider
        $this->app->register('Modules\\' . $module->getName() . '\Providers\RouteServiceProvider');
    }
}
