<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use App\Models\Setting; // Import your Setting model
use Illuminate\Support\Facades\File; // Import File facade
use Illuminate\Support\Facades\Crypt; // Import Crypt facade for decryption
use Illuminate\Support\Facades\Log; // Optional: for logging decryption errors

class MailConfigServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Check if the application is installed before proceeding
        $installFlagPath = storage_path('app/installed.flag');
        if (!File::exists($installFlagPath) && !$this->app->runningInConsole()) {
            // Application is not yet installed, and we are not in a console command
            // Log::debug('[MailConfigServiceProvider] boot: Skipped, application not installed.'); // Optional: log the skip
            // Log::debug('[MailConfigServiceProvider] Running in console, skipping dynamic mail config from DB.');
            return;
        }

        try {
            // Get the mailer type from settings, default to current config or 'smtp'
            $mailer = Setting::getValue('mail_mailer', Config::get('mail.default', 'smtp'));

            // Only proceed if a mailer is determined (either from settings or default config)
            if ($mailer) {
                Config::set('mail.default', $mailer);
                Log::debug("[MailConfigServiceProvider] Mailer set to '{$mailer}'.");

                // Set common 'from' address and name
                $fromAddress = Setting::getValue('mail_from_address', Config::get('mail.from.address'));
                $fromName = Setting::getValue('mail_from_name', Config::get('mail.from.name'));

                if ($fromAddress) Config::set('mail.from.address', $fromAddress);
                if ($fromName) Config::set('mail.from.name', $fromName);

                // Configure the specific mailer (e.g., smtp)
                $mailerConfigPath = "mail.mailers.{$mailer}";

                // Helper to get setting or fallback to existing config for the specific mailer
                $getMailerSetting = function (string $key, $configDefault = null) use ($mailerConfigPath) {
                    return Setting::getValue($key, Config::get("{$mailerConfigPath}.{$configDefault}", null));
                };

                if ($host = $getMailerSetting('mail_host', 'host')) Config::set("{$mailerConfigPath}.host", $host);
                if ($port = $getMailerSetting('mail_port', 'port')) Config::set("{$mailerConfigPath}.port", $port);
                if ($username = $getMailerSetting('mail_username', 'username')) Config::set("{$mailerConfigPath}.username", $username);

                $dbPassword = Setting::getValue('mail_password'); // Get raw value from DB
                if ($dbPassword !== null) { // Check if the setting exists, even if empty
                    try {
                        // Attempt to decrypt only if it's not an empty string, otherwise set password to empty
                        $decryptedPassword = !empty($dbPassword) ? Crypt::decryptString($dbPassword) : '';
                        Config::set("{$mailerConfigPath}.password", $decryptedPassword);
                    } catch (\Illuminate\Contracts\Encryption\DecryptException $e) {
                        Log::warning("[MailConfigServiceProvider] Could not decrypt mail_password. Using raw value if non-empty, or empty if it was. Error: " . $e->getMessage());
                        // If decryption fails on a non-empty string, use the raw value.
                        // If $dbPassword was empty, it would have been handled by the ternary above.
                        Config::set("{$mailerConfigPath}.password", $dbPassword);
                    }
                }

                if ($encryption = $getMailerSetting('mail_encryption', 'encryption')) Config::set("{$mailerConfigPath}.encryption", $encryption);
                Log::debug("[MailConfigServiceProvider] Dynamic mail configuration applied for mailer '{$mailer}'.");
            }
        } catch (\Exception $e) {
            Log::error('[MailConfigServiceProvider] Error during dynamic mail configuration: ' . $e->getMessage());
        }
    }
}
