<?php

namespace App\Providers;

use App\Actions\Jetstream\DeleteUser;
use Illuminate\Support\Facades\Config;
use Laravel\Jetstream\Features;
use Illuminate\Support\Facades\File; // Import File facade
use Illuminate\Support\Facades\Vite;
use Illuminate\Support\ServiceProvider;
use Laravel\Jetstream\Jetstream;
use App\Models\Team;
use App\Models\Setting;
use App\Models\Membership;
use Illuminate\Support\Facades\Log;

class JetstreamServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        // Bind your custom Jetstream actions using the application service container.
        // Using singleton is often appropriate for action classes.
        $this->app->singleton(
            \Laravel\Jetstream\Contracts\CreatesTeams::class,
            \App\Actions\Jetstream\CreateTeam::class
        );
        // If you have other custom actions, bind them here as well:
        // $this->app->singleton(\Laravel\Jetstream\Contracts\UpdatesTeamNames::class, \App\Actions\Jetstream\UpdateTeamName::class);
        // $this->app->singleton(\Laravel\Jetstream\Contracts\AddsTeamMembers::class, \App\Actions\Jetstream\AddTeamMember::class);
        // ... and so on for other actions you've customized.
    }
    
    public function boot(): void
    {
        // Check if the application is installed before proceeding
        $installFlagPath = storage_path('app/installed.flag');
        if (!File::exists($installFlagPath) && !$this->app->runningInConsole()) {
            // Application is not yet installed, and we are not in a console command
            // Log::debug('[JetstreamServiceProvider] boot: Skipped, application not installed.'); // Optional: log the skip
            return; // Exit the boot method early
        }

        $this->configurePermissions();

        Jetstream::deleteUsersUsing(DeleteUser::class);
        Jetstream::useTeamModel(Team::class);
        Jetstream::useMembershipModel(Membership::class);

        Vite::prefetch(concurrency: 3);

        try {
            if (!$this->app->runningInConsole()) {
                $allowTeams = Setting::getValue('allow_user_team_access', '0') === '1';
                // Log::debug('[JetstreamServiceProvider] boot: allow_user_team_access setting: ' . ($allowTeams ? 'Enabled' : 'Disabled'));

                $currentFeatures = Config::get('jetstream.features', []);
                // Log::debug('[JetstreamServiceProvider] boot: Features before modification: ', $currentFeatures);

                // Remove Features::teams() if it exists, to ensure DB setting is authoritative
                $filteredFeatures = array_filter($currentFeatures, function ($feature) {
                    // Check if the feature is the Teams feature, considering it might be an array [Features::teams(), configArray]
                    return !($feature === Features::teams() || (is_array($feature) && isset($feature[0]) && $feature[0] === Features::teams()[0]));
                });

                if ($allowTeams) {
                    $filteredFeatures[] = Features::teams(['invitations' => true]);
                }
                Config::set('jetstream.features', array_values(array_unique($filteredFeatures, SORT_REGULAR)));
                // Log::debug('[JetstreamServiceProvider] boot: Features after modification: ', Config::get('jetstream.features'));
            }
        } catch (\Exception $e) {
            Log::warning('JetstreamServiceProvider (boot): Could not dynamically configure Jetstream team features due to an error: ' . $e->getMessage());
        }
    }
    protected function configurePermissions(): void
    {
        Jetstream::defaultApiTokenPermissions(['read']);

        Jetstream::permissions([
            'create',
            'read',
            'update',
            'delete',
        ]);
    }
}
