<?php

namespace App\Notifications; // Adjust namespace if you put it in a module

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Modules\BusinessDirectory\Entities\Review; // Adjust if Review model is elsewhere

class NewReviewForApprovalNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public Review $review;

    /**
     * Create a new notification instance.
     */
    public function __construct(Review $review)
    {
        $this->review = $review;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database']; // Add 'database' for in-app notifications
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $companyName = $this->review->company->name ?? 'a company';
        $userName = $this->review->user->name ?? 'A user';
        $reviewUrl = route('admin.businessdirectory.reviews.show', $this->review->id); // Ensure this route exists

        return (new MailMessage)
                    ->subject('New Review Submitted for Approval: ' . $companyName)
                    ->greeting('Hello ' . $notifiable->name . ',')
                    ->line("A new review has been submitted by {$userName} for {$companyName}.")
                    ->line("Review Title: " . ($this->review->title ?? 'N/A'))
                    ->line("Rating: " . $this->review->rating . "/5")
                    ->action('View Review', $reviewUrl)
                    ->line('Please review and approve or reject it at your earliest convenience.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'review_id' => $this->review->id,
            'company_name' => $this->review->company->name ?? 'N/A',
            'user_name' => $this->review->user->name ?? 'Anonymous',
            'message' => "New review for {$this->review->company->name} needs approval.",
            'link' => route('admin.businessdirectory.reviews.show', $this->review->id), // Link for in-app notification
        ];
    }
}
