<?php

namespace App\Notifications;

use App\Models\User; // To type-hint the installer
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Nwidart\Modules\Module; // To type-hint the installed module

class ModuleInstalledNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public Module $module;
    public ?User $installer; // The user who installed the module, can be null if system-initiated

    /**
     * Create a new notification instance.
     *
     * @param \Nwidart\Modules\Module $module The module that was installed.
     * @param \App\Models\User|null $installer The user who performed the installation.
     */
    public function __construct(Module $module, ?User $installer = null)
    {
        $this->module = $module;
        $this->installer = $installer;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  object  $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        // By default, send via mail and store in database.
        // You can customize this based on user preferences or notification type.
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  object  $notifiable
     */
    public function toMail(object $notifiable): MailMessage
    {
        $installerName = $this->installer ? $this->installer->name : 'System';
        $subject = "New Module Installed: {$this->module->getName()}";

        return (new MailMessage)
                    ->subject($subject)
                    ->greeting("Hello ". ($notifiable->name ?? 'Admin') .",")
                    ->line("A new module, '{$this->module->getName()}', has been successfully installed.")
                    ->lineIf($this->installer, "Installed by: {$installerName}.")
                    ->line("Module Description: " . $this->module->get('description', 'N/A'))
                    ->action('View Modules', route('admin.modules.index')) // Assumes this route exists
                    ->line('You can manage this module from the admin panel.');
    }

    /**
     * Get the array representation of the notification.
     *
     * This is used for the 'database' channel.
     *
     * @param  object  $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'module_name' => $this->module->getName(),
            'module_alias' => $this->module->getAlias(), // e.g., lower-name version
            'module_description' => $this->module->get('description', 'N/A'),
            'status_message' => "Module '{$this->module->getName()}' has been installed.",
            'installer_id' => $this->installer ? $this->installer->id : null,
            'installer_name' => $this->installer ? $this->installer->name : 'System',
            'action_url' => route('admin.modules.index'), // For linking in the UI
            'icon' => 'fas fa-puzzle-piece', // Example: Font Awesome icon class
        ];
    }
}
