<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory; // Keep this if you have a UserFactory
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Laravel\Jetstream\HasProfilePhoto;
use Laravel\Sanctum\HasApiTokens;
use Laravel\Jetstream\HasTeams; // Import the HasTeams trait
use Modules\BusinessDirectory\Entities\Company; // Corrected namespace
// use Modules\BusinessDirectory\Entities\Company as BusinessDirectoryCompany; // Use an alias if Company model name clashes
use App\Models\WalletWithdrawalRequest; // Add this line
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne; // Added for personalTeam
use Spatie\Permission\Traits\HasRoles; // Import the Spatie HasRoles trait
use Illuminate\Database\Eloquent\Model as EloquentModel; // Alias for related model type hint
use Illuminate\Support\Facades\Log; // Import the Log facade

class User extends Authenticatable
{
    use HasApiTokens;
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory; // Ensure you have a UserFactory at database/factories/UserFactory.php
    use HasProfilePhoto;
    use Notifiable;
    use TwoFactorAuthenticatable;
    use HasTeams; // Use the HasTeams trait
    use HasRoles; // Use the Spatie HasRoles trait

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',                 // For simple role management
        'is_admin',             // Boolean flag for admin status
        'suspended_at',         // Timestamp for user suspension
        'credit_balance',       // For Credits System
        'wallet_balance',       // For Wallet System
        'provider_name',        // For socialite provider
        'provider_id',          // For socialite provider ID
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array<int, string>
     */
    protected $appends = [
        'profile_photo_url',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'suspended_at' => 'datetime', // Cast suspended_at to datetime
            'credit_balance' => 'integer',
            'wallet_balance' => 'decimal:2',
        ];
    }

    /**
     * Check if the user is an admin (admin or super_admin).
     *
     * @return bool
     */
    public function isAdmin(): bool
    {
        return $this->is_admin; // Directly check the boolean attribute
    }

    /**
     * Check if the user is a super admin.
     *
     * @return bool
     */
    public function isSuperAdmin(): bool
    {
        return $this->role === 'super_admin';
    }

    // The hasRole method is provided by Spatie\Permission\Traits\HasRoles
    // If you are using the 'role' column for simple role management and not Spatie's full system,
    // then your existing hasRole method is fine. If you intend to use Spatie's roles and permissions,
    // you would typically remove this custom hasRole method and rely on Spatie's.
    // public function hasRole(string $roleName): bool
    // {
    //     // This is for the simple 'role' column.
    //     // If using Spatie, this method might conflict or be redundant.
    //     return $this->role === $roleName;
    // }

    /**
     * Get all of the subscriptions for the user.
     */
    public function subscriptions(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Get the user's current active (or trialing) subscription.
     *
     * @return \App\Models\Subscription|null
     */
    public function currentSubscription(): ?Subscription
    {
        return $this->subscriptions()
                    ->whereIn('status', ['active', 'trialing'])
                    ->orderBy('created_at', 'desc')
                    ->first();
    }

    /**
     * Get all companies owned by the user.
     */
    public function companies(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Company::class, 'user_id'); // Ensure Modules\BusinessDirectory\Models\Company is correct
    }

    /**
     * Get all of the credit transactions for the user.
     */
    public function creditTransactions(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(UserCreditTransaction::class)->orderBy('created_at', 'desc');
    }

    /**
     * Get all of the wallet transactions for the user.
     */
    public function walletTransactions(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(WalletTransaction::class)->orderBy('created_at', 'desc');
    }

    /**
     * Get all of the tickets for the user.
     */
    public function tickets(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Ticket::class); // Assuming you have a Ticket model
    }

    /**
     * Get all of the wallet withdrawal requests for the user.
     */
    public function walletWithdrawalRequests(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(WalletWithdrawalRequest::class)->orderBy('created_at', 'desc');
    }

    /**
     * Accessor for the user's current credit balance.
     *
     * @return int
     */
    public function getCurrentCreditBalanceAttribute(): int
    {
        return (int) $this->credit_balance; // Assumes 'credit_balance' is the column name
    }

    /**
     * Deduct credits from the user's balance and log the transaction.
     *
     * @param int $amountToDeduct The positive amount of credits to deduct.
     * @param string $transactionType A specific type for the transaction (e.g., 'spend_feature_xyz').
     * @param string $description Description of the transaction.
     * @param EloquentModel|null $relatedModel Optional related model for the transaction.
     * @return bool True on success, false on failure (e.g., insufficient credits).
     */
    public function deductCredits(int $amountToDeduct, string $transactionType, string $description, ?EloquentModel $relatedModel = null): bool
    {
        if ($amountToDeduct <= 0) {
            Log::warning("Attempted to deduct non-positive credit amount ({$amountToDeduct}) for user {$this->id}.");
            return false;
        }

        if ($this->credit_balance < $amountToDeduct) {
            Log::warning("User {$this->id} - Insufficient credits for deduction. Required: {$amountToDeduct}, Available: {$this->credit_balance}. Description: {$description}");
            return false;
        }

        $this->credit_balance -= $amountToDeduct;

        if ($this->save()) {
            UserCreditTransaction::create([
                'user_id' => $this->id,
                'type' => $transactionType,
                'credits_changed' => -$amountToDeduct, // Store as negative for spending
                'balance_after_transaction' => $this->credit_balance,
                'description' => $description,
                'related_type' => $relatedModel ? get_class($relatedModel) : null,
                'related_id' => $relatedModel ? $relatedModel->id : null,
            ]);
            Log::info("User {$this->id} - Deducted {$amountToDeduct} credits. New balance: {$this->credit_balance}. Type: {$transactionType}. Description: {$description}");
            return true;
        }
        Log::error("User {$this->id} - Failed to save user model after credit deduction. Transaction not logged.");
        return false;
    }

    /**
     * Award credits to the user's balance and log the transaction.
     *
     * @param int $amountToAward The positive amount of credits to award.
     * @param string $transactionType A specific type for the transaction (e.g., 'award_purchase', 'admin_grant').
     * @param string $description Description of the transaction.
     * @param EloquentModel|null $relatedModel Optional related model for the transaction.
     * @return bool True on success, false on failure.
     */
    public function awardCredits(int $amountToAward, string $transactionType, string $description, ?EloquentModel $relatedModel = null): bool
    {
        if ($amountToAward <= 0) {
            Log::warning("Attempted to award non-positive credit amount ({$amountToAward}) for user {$this->id}.");
            return false;
        }

        $this->credit_balance += $amountToAward;
        // Similar to deductCredits, save the user and then create a UserCreditTransaction
        // with a positive 'credits_changed' value.
        // For brevity, only showing the balance update and save.
        // You would add the UserCreditTransaction::create(...) call here as well.
        return $this->save(); // Placeholder - ensure transaction logging is added
    }
}
