<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany; // Added for relationship


class SubscriptionPlan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'price',
        'currency',
        'interval',
        'interval_count',
        'trial_period_days',
        'features',
        'active',
        'paypal_plan_id',
        'braintree_plan_id', // Added
        'resource_limits',
        'stripe_product_id',
        'stripe_price_id',
        'paystack_plan_code',
        'custom_gateway_plan_id', // Changed from authorizenet_plan_id to match schema
        'target_role', // Add this
        'available_gateways',
        'credits_awarded_on_purchase',
        'credits_awarded_on_renewal',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'features' => 'array',
        'resource_limits' => 'array',
        'active' => 'boolean',
        'available_gateways' => 'array',
        'trial_period_days' => 'integer', // Added
        'interval_count' => 'integer',    // Added
        'credits_awarded_on_purchase' => 'integer',
        'credits_awarded_on_renewal' => 'integer',
    ];

    /**
     * Get the subscriptions associated with the plan.
     */
    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName(): string
    {
        return 'slug'; // Use slug for route model binding
    }

    /**
     * Check if the plan has a specific feature.
     *
     * @param string $featureKey The key of the feature to check.
     * @return bool
     */
    public function hasFeature(string $featureKey): bool
    {
        if (empty($this->features)) {
            return false;
        }

        foreach ($this->features as $feature) {
            if (isset($feature['key']) && $feature['key'] === $featureKey) {
                return true; // Feature key exists in the plan's features array
            }
        }
        return false;
    }

    /**
     * Get the credit cost for a specific feature key.
     * It will look up the cost from the FeatureCreditCost model.
     *
     * @param string $featureKey
     * @return float|null Returns the credit cost or null if not defined.
     */
    public function getFeatureCreditCost(string $featureKey): ?float
    {
        // Assuming FeatureCreditCost model exists and is correctly namespaced
        // and has 'feature_key' and 'credit_cost' columns.
        $costEntry = \App\Models\FeatureCreditCost::where('feature_key', $featureKey)->first(); // Use the correct namespace

        return $costEntry ? (float) $costEntry->credit_cost : null;
    }
}
