<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Subscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'subscription_plan_id',
        'payment_gateway',
        'gateway_subscription_id',
        'gateway_transaction_id', // For one-off transaction IDs if applicable
        'status',
        'trial_ends_at',
        'starts_at',
        'ends_at',
        'cancelled_at',
        'price_at_purchase',
        'currency_at_purchase',
        'notes', // Add this
    ];

    protected $casts = [
        'trial_ends_at' => 'datetime',
        'starts_at' => 'datetime',
        'ends_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'price_at_purchase' => 'decimal:2', // Optional: for explicitness
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function plan(): BelongsTo
    {
        return $this->belongsTo(SubscriptionPlan::class, 'subscription_plan_id');
    }

    /**
     * Update subscription details from Paystack transaction data.
     *
     * @param array $paystackData The 'data' array from Paystack verification response.
     * @return bool
     */
    public function updateFromPaystackData(array $paystackData): bool
    {
        $plan = $this->plan; // Get the associated plan

        if (!$plan) {
            Log::error("Subscription model: Plan not found for subscription ID {$this->id}");
            return false;
        }

        $this->gateway_transaction_id = $paystackData['reference'] ?? $this->gateway_transaction_id;
        // $this->status = $plan->trial_period_days > 0 ? 'trialing' : 'active'; // This will be set by the calling controller or webhook

        // If starts_at is not already set (e.g., by a pending record or previous activation), set it to now.
        // This ensures that if a webhook activates a subscription that wasn't fully processed by a callback,
        // it still gets a start date.
        if (!$this->starts_at) {
            $this->starts_at = now();
        }

        // Determine the billing cycle anchor (either end of trial or start of subscription)
        $billingCycleAnchor = $this->starts_at;
        if ($this->status === 'trialing' && $plan->trial_period_days > 0) {
            $this->trial_ends_at = $this->starts_at->copy()->addDays($plan->trial_period_days);
            $billingCycleAnchor = $this->trial_ends_at;
        } elseif ($this->status === 'active' && $plan->trial_period_days > 0 && $this->trial_ends_at && $this->trial_ends_at->isFuture()) {
            // If it became active but trial_ends_at was already set and is in the future (e.g. PayPal trial)
            $billingCycleAnchor = $this->trial_ends_at;
        }

        // Calculate ends_at based on the plan's interval and interval_count
        if ($plan->interval && $plan->interval_count) {
            $this->ends_at = $billingCycleAnchor->copy()->add($plan->interval, $plan->interval_count);
        } else {
            // If interval or interval_count is missing, log an error and potentially set a default or null.
            Log::error("Subscription model: Plan ID {$plan->id} is missing interval or interval_count. Cannot calculate ends_at for subscription ID {$this->id}.");
            // $this->ends_at = null; // Or a far future date, or handle as per business logic
        }

        return $this->save();
    }

    /**
     * Update subscription details from PayPal subscription data.
     *
     * @param object $paypalSubscriptionResource The resource object from PayPal API.
     * @return bool
     */
    public function updateFromPaypalData(object $paypalSubscriptionResource): bool
    {
        $plan = $this->plan;
        if (!$plan) {
            Log::error("Subscription model: Plan not found for subscription ID {$this->id} while updating from PayPal data.");
            return false;
        }

        // The gateway_subscription_id should already match $paypalSubscriptionResource->id
        // $this->status = strtolower($paypalSubscriptionResource->status); // Status is set by the calling controller/webhook

        // PayPal's start_time is when the subscription agreement was created or became active.
        if (isset($paypalSubscriptionResource->start_time) && !$this->starts_at) {
            $this->starts_at = \Carbon\Carbon::parse($paypalSubscriptionResource->start_time);
        } elseif (!$this->starts_at) {
            $this->starts_at = now();
        }

        // Determine billing cycle anchor
        $billingCycleAnchor = $this->starts_at;

        // Check for trial information from PayPal if available (this part is highly dependent on PayPal's response structure for subscriptions)
        // For example, if PayPal provides billing_info.cycle_executions and one is a trial:
        if (isset($paypalSubscriptionResource->billing_info, $paypalSubscriptionResource->billing_info->cycle_executions)) {
            foreach ($paypalSubscriptionResource->billing_info->cycle_executions as $cycle) {
                if (strtoupper($cycle->tenure_type) === 'TRIAL' && isset($cycle->cycles_remaining) && $cycle->cycles_remaining > 0 && isset($cycle->total_cycles) && $cycle->total_cycles > 0) {
                    // This is a rough estimation. PayPal's `next_billing_time` is more reliable for the end of the current cycle (trial or regular).
                    if ($plan->trial_period_days > 0) { // Use local plan's trial days as a fallback or primary
                        $this->trial_ends_at = $this->starts_at->copy()->addDays($plan->trial_period_days);
                        $billingCycleAnchor = $this->trial_ends_at;
                    }
                    break;
                }
            }
        } elseif ($this->status === 'trialing' && $plan->trial_period_days > 0) { // Fallback to local plan trial if PayPal doesn't specify
            $this->trial_ends_at = $this->starts_at->copy()->addDays($plan->trial_period_days);
            $billingCycleAnchor = $this->trial_ends_at;
        }

        // PayPal's `billing_info->next_billing_time` is often the most reliable source for the next payment date,
        // which can be considered the `ends_at` for the current cycle.
        if (isset($paypalSubscriptionResource->billing_info, $paypalSubscriptionResource->billing_info->next_billing_time)) {
            $this->ends_at = \Carbon\Carbon::parse($paypalSubscriptionResource->billing_info->next_billing_time);
        } elseif ($plan->interval && $plan->interval_count) { // Fallback to local plan interval calculation
            $this->ends_at = $billingCycleAnchor->copy()->add($plan->interval, $plan->interval_count);
        } else {
            Log::error("Subscription model: Plan ID {$plan->id} is missing interval or interval_count. Cannot calculate ends_at for PayPal subscription ID {$this->id}.");
        }

        return $this->save();
    }
}