<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema; // Import Schema
use Illuminate\Support\Facades\Crypt; // Import Crypt for decryption
use Illuminate\Support\Facades\Log;    // Import Log
use Illuminate\Support\Str; // Import the Str facade

class Setting extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'key',
        'value',
        'type',
        'group',
    ];

    /**
     * Check if the current cache store supports tagging.
     *
     * @return bool
     */
    private static function cacheSupportsTags(): bool
    {
        return Cache::store(config('cache.default'))->getStore() instanceof \Illuminate\Contracts\Cache\TaggableStore;
    }

    /**
     * Get a setting value by its key.
     * Caches the value for performance.
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function getValue(string $key, $default = null)
    {
        $cacheKey = 'setting_' . $key;
        $cacheTags = ['settings', 'setting:' . $key];
        $supportsTags = self::cacheSupportsTags();

        if ($supportsTags) {
            if (Cache::tags($cacheTags)->has($cacheKey)) {
                return Cache::tags($cacheTags)->get($cacheKey);
            }
        } elseif (Cache::has($cacheKey)) {
            return Cache::get($cacheKey);
        }

        try {
            $connectionName = config('database.default');
            if (config("database.connections.{$connectionName}.database") && Schema::connection($connectionName)->hasTable('settings')) {
                $setting = static::on($connectionName)->where('key', $key)->first();
                if ($setting) {
                    // Handle decryption for 'password' type settings
                    if ($setting->type === 'password' && !empty($setting->value)) {
                        try {
                            $decryptedValue = Crypt::decryptString($setting->value);
                            // Cache the decrypted value
                            self::cacheValue($cacheKey, $decryptedValue, $cacheTags, $supportsTags);
                            return $decryptedValue;
                        } catch (\Illuminate\Contracts\Encryption\DecryptException $e) {
                            Log::error("[Setting::getValue] Decryption failed for key '{$key}'. Error: " . $e->getMessage() . ". Ensure APP_KEY is consistent. Returning default.");
                            return $default;
                        }
                    }

                    // Cast the value based on its stored type before caching and returning
                    $typedValue = self::castRetrievedValue($setting->value, $setting->type ?? 'text');
                    self::cacheValue($cacheKey, $typedValue, $cacheTags, $supportsTags);
                    return $typedValue;
                } else {
                    // Setting not found in DB, return default. No need to log this as a debug usually.
                    return $default;
                }
            }
        } catch (\Illuminate\Database\QueryException $e) {
            // This can happen if DB connection is not valid (e.g., pre-install, wrong credentials after install but before fix)
            Log::warning("Setting::getValue: Database query failed for key '{$key}'. Returning default. Error: " . $e->getMessage());
        } catch (\Exception $e) {
            // Catch any other unexpected errors
            Log::error("Setting::getValue: Unexpected error for key '{$key}'. Returning default. Error: " . $e->getMessage());
        }

        return $default; // Return default if DB not ready or setting not found
    }

    /**
     * Helper method to cache a value.
     * @param string $cacheKey
     * @param mixed $value
     * @param array $cacheTags
     * @param bool $supportsTags
     */
    private static function cacheValue(string $cacheKey, $value, array $cacheTags, bool $supportsTags): void
    {
        $cacheMethod = $supportsTags ? Cache::tags($cacheTags) : Cache::store();
        $cacheMethod->forever($cacheKey, $value);
    }

    /**
     * Set a setting value by its key.
     */
public static function setValue(string $key, $value, string $name = null, string $group = 'Application', string $type = 'text'): bool
    {
        try {
            $cacheKey = 'setting_' . $key;
            $cacheTags = ['settings', 'setting:' . $key];
            $supportsTags = self::cacheSupportsTags();

            $connectionName = config('database.default');
            if (config("database.connections.{$connectionName}.database") && Schema::connection($connectionName)->hasTable('settings')) { // Ensure table exists
                $storableValue = $value;
                switch ($type) {
                    case 'boolean':
                        $storableValue = filter_var($value, FILTER_VALIDATE_BOOLEAN) ? '1' : '0';
                        break;
                    case 'password':
                        // Encrypt the value if it's a password type and not empty
                        $storableValue = !empty($value) ? Crypt::encryptString($value) : null;
                        break;
                    case 'json':
                        $storableValue = is_array($value) || is_object($value) ? json_encode($value) : $value;
                        break;
                    // Add other pre-storage casting if needed
                }

                static::on($connectionName)->updateOrCreate(
                    ['key' => $key],
                    [
                        'name' => $name ?? \Illuminate\Support\Str::title(str_replace('_', ' ', $key)),
                        'value' => $storableValue, // This will be the encrypted value for passwords
                        'group' => $group,
                        'type' => $type
                    ]
                );
                if ($supportsTags) {
                    Cache::tags($cacheTags)->forget($cacheKey); // Bust cache for this specific key using tags
                } else {
                    Cache::forget($cacheKey);
                }
                return true; // Indicate success
            } else {
                Log::error("Setting::setValue: 'settings' table not found or DB not configured. Cannot set key '{$key}'.");
                return false; // Indicate failure
            }
        } catch (\Exception $e) {
            Log::error("Setting::setValue: Failed to set key '{$key}'. Error: " . $e->getMessage());
            return false; // Indicate failure
        }
    }

    /**
     * Casts the retrieved string value from the database to its correct PHP type.
     *
     * @param string|null $value
     * @param string $type
     * @return mixed
     */
    private static function castRetrievedValue(?string $value, string $type)
    {
        if ($value === null) {
            return null;
        }
        switch ($type) {
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            case 'number':
                return is_numeric($value) ? ($value + 0) : $value; // Converts to int/float
            case 'json':
                $decoded = json_decode($value, true);
                return json_last_error() === JSON_ERROR_NONE ? $decoded : $value; // Return array/object or original string if not valid JSON
            default:
                return $value;
        }
    }
}
