<?php

namespace App\Jobs;

use App\Models\Newsletter;
use App\Models\NewsletterSubscription;
use App\Mail\NewsletterMail;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class SendNewsletterJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public Newsletter $newsletter;

    /**
     * Create a new job instance.
     */
    public function __construct(Newsletter $newsletter)
    {
        $this->newsletter = $newsletter;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        if ($this->newsletter->status !== 'sending') {
            Log::warning("SendNewsletterJob: Newsletter ID {$this->newsletter->id} is not in 'sending' status. Current status: {$this->newsletter->status}. Aborting job.");
            return;
        }

        Log::info("SendNewsletterJob: Starting to send newsletter ID {$this->newsletter->id} ('{$this->newsletter->subject}').");

        $subscribers = NewsletterSubscription::where('status', 'subscribed')->cursor(); // Use cursor for memory efficiency
        $sentCount = 0;
        $failedCount = 0;

        foreach ($subscribers as $subscriber) {
            try {
                Mail::to($subscriber->email)->send(new NewsletterMail($this->newsletter->subject, $this->newsletter->content));
                $sentCount++;
                // Optional: Add a small delay if your mail provider has rate limits
                // sleep(1); 
            } catch (\Exception $e) {
                $failedCount++;
                Log::error("SendNewsletterJob: Failed to send newsletter to {$subscriber->email} for newsletter ID {$this->newsletter->id}. Error: " . $e->getMessage());
            }
        }

        $finalStatus = ($failedCount > 0 && $sentCount === 0) ? 'failed' : 'sent';
        $this->newsletter->update([
            'status' => $finalStatus,
            'sent_at' => now(),
            // recipients_count was set when dispatching, but you could update it here if needed
        ]);

        Log::info("SendNewsletterJob: Finished sending newsletter ID {$this->newsletter->id}. Sent: {$sentCount}, Failed: {$failedCount}. Status set to: {$finalStatus}.");
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("SendNewsletterJob: Job failed for newsletter ID {$this->newsletter->id}. Error: " . $exception->getMessage());
        $this->newsletter->update(['status' => 'failed']);
    }
}
