<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\ModulePermission; // To check permissions for the 'admin' role
use Nwidart\Modules\Facades\Module; // To verify module existence and status
use Symfony\Component\HttpFoundation\Response;

class CheckAdminModuleAccess
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     * @param  string  $moduleName The canonical name of the module (e.g., "Blog", "BusinessDirectory")
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function handle(Request $request, Closure $next, string $moduleName): Response
    {
        $user = Auth::user();

        // First, ensure the user is an admin (this also implies authenticated)
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized action. Admin access required.');
        }

        // Super Admins have access to all modules
        if ($user->isSuperAdmin()) {
            return $next($request);
        }

        // For regular 'admin' role, check specific module permission
        $hasPermission = ModulePermission::where('role_name', 'admin')
                                         ->where('module_name', $moduleName)
                                         ->exists();

        if (!$hasPermission) {
            abort(403, "Unauthorized action. You do not have permission to access the {$moduleName} module management.");
        }

        return $next($request);
    }
}
