<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Services\WalletService;
use App\Models\WalletWithdrawalRequest;
use Illuminate\Support\Facades\Log;
use Nwidart\Modules\Facades\Module; // For listing payment gateways
use Illuminate\Support\Str;

class WalletController extends Controller
{
    protected WalletService $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
        $this->middleware('auth');
    }

    /**
     * Show the form for users to deposit funds into their wallet.
     */
    public function showDepositForm()
    {
        if (function_exists('setting') && setting('wallet_system_enabled', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Wallet system is currently disabled.');
        }
        if (function_exists('setting') && setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Wallet deposits are currently disabled.');
        }

        $user = Auth::user();
        $currentBalance = $this->walletService->getUserWalletBalance($user);

        // Fetch available payment gateways (similar to SubscriptionController)
        $paymentGateways = [];
        $user = Auth::user(); // Assuming user is authenticated

        if (class_exists(\Nwidart\Modules\Facades\Module::class)) {
            foreach (Module::allEnabled() as $module) {
                Log::debug("WalletController@showDepositForm: Checking module: " . $module->getName());

                if ($module->get('type') !== 'payment_gateway' && $module->get('category') !== 'payment_gateway') {
                    if ($module->getName() === 'PaynowGateway') { // Example specific log
                        Log::debug("PaynowGateway module skipped for wallet: Not identified as payment_gateway type/category.", [
                            'type' => $module->get('type'),
                            'category' => $module->get('category')
                        ]);
                    }
                    continue;
                }

                $moduleGatewayStoreKey = $module->get('gateway_key', Str::snake($module->getName()));
                Log::debug("WalletController@showDepositForm: Module " . $module->getName() . " - gateway_key: " . $moduleGatewayStoreKey);

                $actualEnabledSettingKey = $module->getLowerName() . '_enabled';
                $isGatewayEnabledInSettings = function_exists('setting') ? setting($actualEnabledSettingKey, '0') : '0';
                Log::debug("WalletController@showDepositForm: Module " . $module->getName() . " - Checking setting '{$actualEnabledSettingKey}': Value is '{$isGatewayEnabledInSettings}'.");

                if ($isGatewayEnabledInSettings == '1') {
                    // Optional: Check if this gateway is specifically allowed for wallet deposits
                    $allowWalletDepositSettingKey = $module->getLowerName() . '_allow_wallet_deposits';
                    // Default to '1' (true) if the specific _allow_wallet_deposits setting doesn't exist,
                    // meaning if the gateway is enabled, it's allowed for deposits unless explicitly disallowed.
                    $isAllowedForWallet = function_exists('setting') ? setting($allowWalletDepositSettingKey, '1') : '1';
                    Log::debug("WalletController@showDepositForm: Module " . $module->getName() . " - Checking wallet deposit allowance '{$allowWalletDepositSettingKey}': Value is '{$isAllowedForWallet}'.");

                    if ($isAllowedForWallet == '1') {
                        $gatewayData = [
                            'name' => $module->get('display_name', $module->getName()), // Use display_name from module.json
                            'key' => $moduleGatewayStoreKey,
                        ];

                        if ($moduleGatewayStoreKey === 'authorizenetgateway') { // Use $moduleGatewayStoreKey for consistency
                            $gatewayData['apiLoginId'] = setting('authorizenet_login_id');
                            $gatewayData['publicClientKey'] = setting('authorizenet_public_client_key');
                        }
                        // Add Stripe publishable key if Stripe module
                        // if ($moduleGatewayStoreKey === 'stripegateway') {
                        //     $gatewayData['publishableKey'] = setting('stripe_publishable_key');
                        // }
                        $paymentGateways[$moduleGatewayStoreKey] = $gatewayData;
                        Log::info("WalletController@showDepositForm: Added gateway " . $module->getName() . " to wallet deposit options.");
                    } elseif ($module->getName() === 'PaynowGateway') { // Example specific log
                        Log::debug("PaynowGateway module skipped for wallet: Not allowed for wallet deposits via key '{$allowWalletDepositSettingKey}'. Value was '{$isAllowedForWallet}'.");
                    }
                } elseif ($module->getName() === 'PaynowGateway') { // Example specific log
                    Log::debug("PaynowGateway module skipped for wallet: Not enabled in settings via key '{$actualEnabledSettingKey}'. Value was '{$isGatewayEnabledInSettings}'.");
                }
            }
        }

        // Optionally, add a "Custom/Bank Transfer" if enabled and configured for deposits
        if (function_exists('setting') && setting('custom_payment_gateway_enabled', '0') == '1' && setting('custom_payment_for_deposits_enabled', '0') == '1') {
            $paymentGateways['custom_deposit'] = [
                'name' => setting('custom_payment_gateway_name', __('Bank Transfer / Other Instructions')),
                'key' => 'custom_deposit', // Differentiate from subscription custom payment if needed
            ];
        }

        // If no gateways are configured for deposit
        if (empty($paymentGateways)) {
            if (Auth::user()->isAdmin()) { // Show a more informative message to admins
                session()->flash('info', 'No payment gateways are configured for wallet deposits. Please configure them in the admin panel.');
            } else {
                session()->flash('error', 'Wallet deposits are temporarily unavailable. Please try again later or contact support.');
            }
        }
        ksort($paymentGateways);

        return view('user.wallet.deposit', compact('currentBalance', 'paymentGateways'));
    }

    /**
     * Process the user's wallet deposit request.
     * For now, this will simulate a successful deposit.
     */
    public function processDeposit(Request $request)
    {
        if (function_exists('setting') && setting('wallet_system_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Wallet deposits are currently disabled.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:1', // Example minimum deposit
            'payment_method' => 'required|string', // Key of the selected payment gateway
        ]);

        $user = Auth::user();
        $amount = (float) $request->input('amount');
        $paymentMethod = $request->input('payment_method');

        // Redirect to the specific gateway's deposit initialization route
        switch ($paymentMethod) {
            case 'authorizenetgateway':
                return redirect()->route('wallet.authorizenet.depositForm', ['amount' => $amount]);
            case 'paypalgateway':
                return redirect()->route('wallet.paypal.initializeDeposit', ['amount' => $amount]);
            case 'paystackgateway':
                return redirect()->route('wallet.paystack.initializeDeposit', ['amount' => $amount]);
            case 'stripegateway':
                return redirect()->route('wallet.stripe.initializeDeposit', ['amount' => $amount]);
            case 'paynow_gateway': // Corrected to match the key used in showDepositForm
                return redirect()->route('wallet.paynow.initializeDeposit', ['amount' => $amount]);
            case 'custom_deposit':
                // Redirect to a page showing bank transfer details or instructions
                // You'll need a route and view for this.
                return redirect()->route('user.wallet.customDepositInstructions', ['amount' => $amount]);
            default:
                Log::warning("Attempted wallet deposit with unknown payment method: {$paymentMethod} by user {$user->id}");
                return redirect()->back()->with('error', 'Invalid payment method selected.')->withInput();
        }

        // The actual deposit record will be created by the gateway's callback/webhook handler
    }

    /**
     * Show the form for users to request a withdrawal from their wallet.
     */
    public function showWithdrawalForm()
    {
        if (function_exists('setting') && setting('wallet_system_enabled', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Wallet system is currently disabled.');
        }
        if (function_exists('setting') && setting('allow_wallet_withdrawals', '0') != '1') {
            return redirect()->route('user.wallet.history')->with('error', 'Wallet withdrawals are currently disabled.');
        }

        $user = Auth::user();
        $currentBalance = $this->walletService->getUserWalletBalance($user);
        $minWithdrawal = (float) setting('wallet_min_withdrawal', '10.00');
        // For now, let's assume bank transfer is the primary method.
        // In the future, you can make withdrawal_methods configurable via settings.
        $withdrawalMethods = [
            'bank_transfer' => 'Bank Transfer',
            // 'paypal' => 'PayPal', // Example for future
        ];

        return view('user.wallet.withdraw_form', compact('currentBalance', 'minWithdrawal', 'withdrawalMethods'));
    }

    /**
     * Process the user's withdrawal request.
     */
    public function submitWithdrawalRequest(Request $request)
    {
        if (function_exists('setting') && setting('wallet_system_enabled', '0') != '1' || setting('allow_wallet_withdrawals', '0') != '1') {
            return redirect()->route('user.wallet.history')->with('error', 'Wallet withdrawals are currently disabled.');
        }

        $user = Auth::user();
        $currentBalance = $this->walletService->getUserWalletBalance($user);
        $minWithdrawal = (float) setting('wallet_min_withdrawal', '10.00');

        $validatedData = $request->validate([
            'amount' => ['required', 'numeric', "min:{$minWithdrawal}", "max:{$currentBalance}"],
            'withdrawal_method' => 'required|string|in:bank_transfer', // Add other methods here if they become available
            // Bank Transfer Specific Fields
            'bank_name' => 'required_if:withdrawal_method,bank_transfer|string|max:255',
            'account_holder_name' => 'required_if:withdrawal_method,bank_transfer|string|max:255',
            'account_number' => 'required_if:withdrawal_method,bank_transfer|string|max:50', // Adjust max as needed
            'branch_code' => 'nullable|string|max:50', // e.g., Swift, IBAN, Sort Code
        ], [
            'amount.max' => 'Withdrawal amount cannot exceed your current balance.',
            'amount.min' => 'Minimum withdrawal amount is ' . $minWithdrawal . ' ' . setting('wallet_default_currency', 'USD') . '.',
        ]);

        $paymentDetails = [];
        if ($validatedData['withdrawal_method'] === 'bank_transfer') {
            $paymentDetails = [
                'bank_name' => $validatedData['bank_name'],
                'account_holder_name' => $validatedData['account_holder_name'],
                'account_number' => $validatedData['account_number'],
                'branch_code' => $validatedData['branch_code'] ?? null,
            ];
        }

        WalletWithdrawalRequest::create([
            'user_id' => $user->id,
            'amount' => $validatedData['amount'],
            'currency' => setting('wallet_default_currency', 'USD'),
            'status' => 'pending',
            'withdrawal_method' => $validatedData['withdrawal_method'],
            'payment_details' => $paymentDetails,
        ]);

        return redirect()->route('user.wallet.history')->with('success', 'Your withdrawal request has been submitted and is pending review.');
    }

    /**
     * Display the specified wallet withdrawal request for the user.
     */
    public function showWithdrawalRequest(WalletWithdrawalRequest $withdrawalRequest)
    {
        if (function_exists('setting') && setting('wallet_system_enabled', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Wallet system is currently disabled.');
        }

        // Ensure the authenticated user owns this withdrawal request
        if ($withdrawalRequest->user_id !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        $statuses = WalletWithdrawalRequest::getStatuses(); // For displaying status label
        return view('user.wallet.withdrawal_request_show', compact('withdrawalRequest', 'statuses'));
    }

    /**
     * Show custom deposit instructions (e.g., for bank transfers).
     */
    public function showCustomDepositInstructions(Request $request)
    {
        if (function_exists('setting') && setting('wallet_system_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Wallet deposits are currently disabled.');
        }
        if (function_exists('setting') && setting('custom_payment_gateway_enabled', '0') != '1' || setting('custom_payment_for_deposits_enabled', '0') != '1') {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'This payment method is currently not available for deposits.');
        }

        $amount = $request->query('amount'); // Get amount from query string if passed
        // Fetch instructions from settings or a dedicated model/table
        $instructions = setting('custom_deposit_instructions', 'Please contact support for deposit instructions.');
        return view('user.wallet.custom_deposit_instructions', compact('instructions', 'amount'));
    }
}