<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\TicketReply;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Notifications\NewTicketReplyNotification; // We'll create this later
use App\Models\User; // For notifying admins

class TicketController extends Controller
{
    public function index()
    {
        $tickets = Auth::user()->tickets()->orderBy('last_reply_at', 'desc')->orderBy('created_at', 'desc')->paginate(10);
        return view('user.tickets.index', compact('tickets'));
    }

    public function create()
    {
        return view('user.tickets.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
            'priority' => 'required|in:low,medium,high',
        ]);

        $ticket = Ticket::create([
            'user_id' => Auth::id(),
            'subject' => $request->subject,
            'message' => $request->message,
            'priority' => $request->priority,
            'status' => 'open', // Or 'pending_admin_reply'
            'last_reply_at' => now(),
        ]);

        // Notify admins (example)
        // $adminUsers = User::where('is_admin', true)->get();
        // foreach ($adminUsers as $admin) {
        //     $admin->notify(new NewTicketNotification($ticket)); // Create this notification
        // }

        return redirect()->route('user.tickets.show', $ticket)->with('success', 'Support ticket created successfully.');
    }

    public function show(Ticket $ticket)
    {
        // Ensure the user owns this ticket
        if ($ticket->user_id !== Auth::id()) {
            abort(403);
        }
        return view('user.tickets.show', compact('ticket'));
    }

    public function reply(Request $request, Ticket $ticket)
    {
        if ($ticket->user_id !== Auth::id()) {
            abort(403);
        }

        if ($ticket->status === 'closed') {
            return back()->with('error', 'This ticket is closed and cannot be replied to.');
        }

        $request->validate([
            'message' => 'required|string|max:5000',
        ]);

        TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
        ]);

        $ticket->update([
            'status' => 'pending_admin_reply', // Or 'open' if you prefer
            'last_reply_at' => now(),
        ]);

        // Notify assigned admin or all admins
        // if ($ticket->assignedAdmin) {
        //     $ticket->assignedAdmin->notify(new NewTicketReplyNotification($ticket, Auth::user()));
        // } else {
        //     $adminUsers = User::where('is_admin', true)->get();
        //     foreach ($adminUsers as $admin) {
        //         $admin->notify(new NewTicketReplyNotification($ticket, Auth::user()));
        //     }
        // }


        return redirect()->route('user.tickets.show', $ticket)->with('success', 'Your reply has been submitted.');
    }
}