<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Nwidart\Modules\Facades\Module;
use Illuminate\Support\Str;

class CreditPurchaseController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the form for users to top up their credits.
     */
    public function showTopupForm()
    {
        if (function_exists('setting') && setting('credits_system_enabled', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Credit system is currently disabled.');
        }
        // You might want a specific setting to enable credit purchases
        // if (function_exists('setting') && setting('allow_credit_purchases', '0') != '1') {
        //     return redirect()->route('dashboard')->with('error', 'Credit purchases are currently disabled.');
        // }

        $user = Auth::user();
        $currentCreditBalance = $user->credit_balance ?? 0;

        $paymentGateways = [];
        if (class_exists(Module::class)) {
            foreach (Module::allEnabled() as $module) {
                Log::debug("CreditPurchaseController@showTopupForm: Checking module: " . $module->getName());

                if ($module->get('type') !== 'payment_gateway' && $module->get('category') !== 'payment_gateway') {
                    continue;
                }

                $moduleGatewayStoreKey = $module->get('gateway_key', Str::snake($module->getName()));
                $actualEnabledSettingKey = $module->getLowerName() . '_enabled';
                $isGatewayEnabledInSettings = function_exists('setting') ? setting($actualEnabledSettingKey, '0') : '0';

                if ($isGatewayEnabledInSettings == '1') {
                    // Optional: Check if this gateway is specifically allowed for credit purchases
                    $allowCreditPurchaseSettingKey = $module->getLowerName() . '_allow_credit_purchases';
                    $isAllowedForCreditPurchase = function_exists('setting') ? setting($allowCreditPurchaseSettingKey, '1') : '1'; // Default to '1' (true)

                    if ($isAllowedForCreditPurchase == '1') {
                        $paymentGateways[$moduleGatewayStoreKey] = [
                            'name' => $module->get('display_name', $module->getName()),
                            'key' => $moduleGatewayStoreKey,
                        ];
                        Log::info("CreditPurchaseController@showTopupForm: Added gateway " . $module->getName() . " to credit top-up options.");
                    }
                }
            }
        }
        ksort($paymentGateways);

        return view('user.credits.topup_form', compact('currentCreditBalance', 'paymentGateways'));
    }

    /**
     * Process the user's credit top-up request.
     */
    public function processTopup(Request $request)
    {
        if (function_exists('setting') && setting('credits_system_enabled', '0') != '1') {
            return redirect()->route('dashboard')->with('error', 'Credit system is currently disabled.');
        }

        $minCredits = setting('credits_min_purchase_amount', 10); // Example: Minimum 10 credits
        $request->validate([
            'credits' => "required|integer|min:{$minCredits}",
            'payment_method' => 'required|string',
        ]);

        $user = Auth::user();
        $creditsToPurchase = (int) $request->input('credits');
        $paymentMethodKey = $request->input('payment_method');

        // Convert credits to an amount if your gateways expect a monetary value
        // This requires a "price per credit" setting.
        // For simplicity, let's assume for now that gateways will handle "credits" as the "amount"
        // or that the gateway-specific controller will do the conversion.
        // $pricePerCredit = (float) setting('price_per_credit', 0.10); // e.g., $0.10 per credit
        // $amountToPay = $creditsToPurchase * $pricePerCredit;

        // Instead of amount, we pass credits. The module controller will handle it.
        $routeParameters = ['credits' => $creditsToPurchase];

        // Construct the route name dynamically: credits.{gateway_key}.initializePurchase
        // e.g., credits.paynow_gateway.initializePurchase
        $routeName = 'credits.' . $paymentMethodKey . '.initializePurchase';

        if (Route::has($routeName)) {
            return redirect()->route($routeName, $routeParameters);
        }

        Log::warning("Attempted credit purchase with unknown payment method or undefined route: {$paymentMethodKey} by user {$user->id}");
        return redirect()->back()->with('error', 'Invalid payment method selected or route not configured.')->withInput();
    }
}