<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Services\CreditService;
use App\Services\WalletService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Nwidart\Modules\Facades\Module; // Ensure Module facade is imported
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;
use Modules\AuthorizeNetGateway\Services\AuthorizeNetService; // For Authorize.Net
use App\Models\User; // Ensure User model is imported

class SubscriptionController extends Controller
{
    protected CreditService $creditService;
    protected WalletService $walletService;

    public function __construct(CreditService $creditService, WalletService $walletService)
    {
        $this->creditService = $creditService;
        $this->walletService = $walletService;
        $this->middleware('auth');
    }
    /**
     */
    public function index()
    {
        if (function_exists('setting') && setting('subscriptions_enabled', '0') != '1') {
            return redirect()->route('dashboard')->with('info', 'The subscription feature is currently not available.');
        }

        $user = Auth::user();
        $currentSubscription = null;
        $userRoles = collect(); // Initialize as an empty collection

        if ($user) {
            $currentSubscription = $user->currentSubscription();
            $userRoles = $user->getRoleNames(); // Get all Spatie roles for the user
        }

        $plans = SubscriptionPlan::where('active', true)
            ->where(function ($query) use ($userRoles) {
                $query->whereNull('target_role'); // Always include general plans
                if ($userRoles->isNotEmpty()) {
                    $query->orWhereIn('target_role', $userRoles->toArray()); // Include plans targeted to any of the user's roles
                }
            })
            ->orderBy('price')->get();
        $moduleDefinitions = [];
        if (class_exists(\Nwidart\Modules\Facades\Module::class)) { // Correct class_exists check for Facade
            foreach (Module::allEnabled() as $module) {
                $moduleName = $module->getName();
                // Assuming module.json has a 'name' or 'display_name' for user-friendly title
                $moduleDisplayName = $module->get('display_name', $module->get('name', $moduleName));
                $definedFeatures = $module->get('features'); // Array of feature definitions from module.json
                $moduleRequiresSubscription = $module->get('requires_subscription', true); // Default to true

                if (is_array($definedFeatures) && !empty($definedFeatures)) {
                    $featuresForThisModuleDef = [];
                    foreach ($definedFeatures as $featureDef) {
                        $featureName = $featureDef['name'] ?? 'unknown_feature';
                        $formFeatureKey = $featureDef['key'] ?? Str::slug($moduleName . ' ' . $featureName);
                        $featuresForThisModuleDef[$formFeatureKey] = [
                            'name' => $featureName,
                            'description' => $featureDef['description'] ?? '',
                            'requires_subscription' => $featureDef['requires_subscription'] ?? $moduleRequiresSubscription,
                            'target_role_hint' => $featureDef['target_role'] ?? null, // For admin UI hints
                        ];
                    }
                    if (!empty($featuresForThisModuleDef)) {
                        $moduleDefinitions[$moduleName] = [
                            'displayName' => $moduleDisplayName,
                            'features' => $featuresForThisModuleDef
                        ];
                    }
                }
            }
        }

        return view('subscriptions.index', compact('plans', 'currentSubscription', 'moduleDefinitions', 'userRoles'));
    }

    public function showCheckout(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        $user = Auth::user();

        if (!$subscriptionPlan->active) {
            return redirect()->route('subscription.plans')->with('error', 'This subscription plan is currently not available.');
        }

        $hasActiveSubscriptionToThisPlan = $user->subscriptions()
            ->where('subscription_plan_id', $subscriptionPlan->id)
            ->whereIn('status', ['active', 'trialing'])
            ->exists();
        if ($hasActiveSubscriptionToThisPlan) {
            return redirect()->route('subscription.plans')->with('info', 'You are already subscribed to this plan.');
        }

        $activeUserSubscription = $user->currentSubscription();
        if ($activeUserSubscription && $activeUserSubscription->subscription_plan_id !== $subscriptionPlan->id) {
            session()->flash('info', 'You are about to switch your subscription plan. Your current plan will be cancelled.');
        }

        $paymentGateways = [];
        $planGateways = $subscriptionPlan->available_gateways ?? []; // Ensure it's an array

        if (is_array($planGateways)) {
            // Handle module-based gateways
            foreach (Module::allEnabled() as $module) {
                Log::debug("SubscriptionController@showCheckout: Checking module: " . $module->getName());
                // Ensure it's a payment gateway module
                if ($module->get('type') !== 'payment_gateway' && $module->get('category') !== 'payment_gateway') {
                    if ($module->getName() === 'PaynowGateway') { // Specific log for Paynow
                        Log::debug("PaynowGateway module skipped: Not identified as payment_gateway type/category.", [
                            'type' => $module->get('type'),
                            'category' => $module->get('category')
                        ]);
                    }
                    continue;
                }

                // This is the key stored in the plan's available_gateways field (e.g., "paypal_gateway")
                $moduleGatewayStoreKey = $module->get('gateway_key', Str::snake($module->getName()));
                Log::debug("SubscriptionController@showCheckout: Module " . $module->getName() . " - gateway_key: " . $moduleGatewayStoreKey);

                if (in_array($moduleGatewayStoreKey, $planGateways)) { // Check if module is allowed for this plan
                   $actualEnabledSettingKey = $module->getLowerName() . '_enabled'; // Convention: lowername_enabled
                    $isGatewayEnabledInSettings = function_exists('setting') ? setting($actualEnabledSettingKey, '0') : '0';
                    Log::debug("SubscriptionController@showCheckout: Module " . $module->getName() . " - Allowed for plan. Checking setting: " . $actualEnabledSettingKey . " - Value: " . $isGatewayEnabledInSettings);

                    if ($isGatewayEnabledInSettings == '1') {
                        $gatewayData = [
                            'name' => $module->get('display_name', $module->getName()),
                            'key' => $moduleGatewayStoreKey,
                        ];

                        // Add specific data for gateways if needed by the view
                        if ($module->getName() === 'AuthorizeNetGateway') { // Use actual module name for comparison
                           $gatewayData['apiLoginId'] = setting('authorizenet_login_id');
                           $gatewayData['publicClientKey'] = setting('authorizenet_public_client_key');
                        }
                        // Add Stripe publishable key if Stripe module
                        // if ($module->getName() === 'StripeGateway') {
                        //     $gatewayData['publishableKey'] = setting('stripe_publishable_key');
                        // }
                        // ... other specific gateway data

                        $paymentGateways[$moduleGatewayStoreKey] = $gatewayData;
                        Log::info("SubscriptionController@showCheckout: Added gateway " . $module->getName() . " to checkout options.");
                    } else {
                        if ($module->getName() === 'PaynowGateway') { // Specific log for Paynow
                           Log::debug("PaynowGateway module skipped: Not enabled in settings via key '{$actualEnabledSettingKey}'. Value was '{$isGatewayEnabledInSettings}'.");
                        }
                    }
                } elseif ($module->getName() === 'PaynowGateway') { // Specific log for Paynow
                    Log::debug("PaynowGateway module skipped: Its gateway_key '{$moduleGatewayStoreKey}' not found in plan's available_gateways.", ['plan_gateways' => $planGateways]);
                }
            }

            // Handle non-module gateways like 'wallet' and 'custom'
            if (in_array('wallet', $planGateways) && function_exists('setting') && setting('wallet_system_enabled', '0') == '1') {
                $paymentGateways['wallet'] = [
                    'name' => __('Pay with Wallet Balance'),
                    'key' => 'wallet',
                    'balance' => $user->wallet_balance ?? 0
                ];
            }
            if (in_array('custom', $planGateways) && function_exists('setting') && setting('custom_payment_gateway_enabled', false)) {
                $paymentGateways['custom'] = [
                    'name' => setting('custom_payment_gateway_name', 'Custom Payment Instructions'),
                    'key' => 'custom',
                ];
            }
        }
        // Ensure consistent order if needed, e.g., by sorting $paymentGateways by key or name
        // ksort($paymentGateways);

        return view('subscriptions.checkout', compact('subscriptionPlan', 'user', 'paymentGateways', 'activeUserSubscription'));
    }

    public function processSubscription(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        $user = Auth::user();
        $paymentMethodInput = $request->input('payment_method'); // Original input, e.g., 'wallet', 'stripe', 'paypal'

        $isNewSubscription = true; // Determine if this is a new sub or renewal (simplified for now)
        $paymentSuccessful = false;
        $gatewayTransactionId = null;
        $errorMessage = 'Subscription payment failed. Please try again.'; // Default error message
        $finalPaymentMethodForRecord = $paymentMethodInput; // For DB record, might be overridden for free plans

        if ($paymentMethodInput === 'wallet') {
            if (function_exists('setting') && setting('wallet_system_enabled', '0') == '1') {
                $walletSpend = $this->walletService->spendFromWallet(
                    $user,
                    $subscriptionPlan->price,
                    'purchase_subscription',
                    "Subscription: {$subscriptionPlan->name}"
                );
                if ($walletSpend) {
                    $paymentSuccessful = true;
                    $gatewayTransactionId = 'WALLET-' . $walletSpend->id;
                } else {
                    $errorMessage = 'Insufficient wallet balance or wallet payment failed.';
                }
            } else {
                $errorMessage = 'Wallet payment method is not enabled.';
            }
        // Handle free plans directly, regardless of selected payment method
        } elseif ($subscriptionPlan->price == 0) {
            Log::info("Processing free subscription for plan: {$subscriptionPlan->name} (ID: {$subscriptionPlan->id}) for user ID: {$user->id}");
            $paymentSuccessful = true;
            $gatewayTransactionId = 'FREE_PLAN-' . Str::upper(Str::random(8));
            $finalPaymentMethodForRecord = 'free'; // Store 'free' as the gateway for record keeping
        } else {
            // Plan is not free, process with the selected payment method input
            if ($paymentMethodInput === 'authorize_net_gateway') {
                $authorizeNetModule = Module::find('AuthorizeNetGateway');
                if ($authorizeNetModule && $authorizeNetModule->isEnabled() && setting('authorizenetgateway_enabled', '0') == '1') {
                    // Redirect to the Authorize.Net module's specific checkout page.
                    // The route name 'subscription.authorizenet.checkoutform' is an assumption
                    // and should match a route defined in Modules/AuthorizeNetGateway/routes/web.php
                    // that points to AuthorizeNetSubscriptionController@showCheckoutForm.
                    Log::info("Redirecting to Authorize.Net checkout form for plan: {$subscriptionPlan->slug}");
                    return redirect()->route('subscription.authorizenet.checkout', ['subscriptionPlan' => $subscriptionPlan->slug]); // Exits early
                } else {
                    // $errorMessage = 'Authorize.Net payment method is currently not available.';
                    // For consistency with other unavailable gateways:
                    return redirect()->back()->with('error', 'Authorize.Net payment method is currently not available.')->withInput(); // Exits early
                }
            } elseif ($paymentMethodInput === 'paypal_gateway') {
                $paypalModule = Module::find('PaypalGateway');
                if ($paypalModule && $paypalModule->isEnabled() && setting('paypalgateway_enabled', '0') == '1') {
                    try {
                        $paypalService = app(\Modules\PaypalGateway\Services\PaypalService::class);
                        // This will throw if paypal_plan_id is missing for this PAID plan
                        $approvalLink = $paypalService->createPaypalSubscription($user, $subscriptionPlan);

                        if ($approvalLink) {
                            // PayPalService::createPaypalSubscription sets 'paypal_subscription_id_pending' in session.
                            $pendingPaypalIdFromSession = session('paypal_subscription_id_pending');

                            if (empty($pendingPaypalIdFromSession)) {
                                Log::error("SubscriptionController: 'paypal_subscription_id_pending' is EMPTY in session after PaypalService call. Cannot create local pending subscription.", ['user_id' => $user->id, 'plan_id' => $subscriptionPlan->id, 'session_dump' => session()->all()]);
                                return redirect()->route('subscription.checkout', ['subscriptionPlan' => $subscriptionPlan->slug])->with('error', 'Failed to initiate PayPal subscription due to a session issue. Please try again.');
                            }

                            // Create the local pending subscription record
                            try {
                                Log::info("SubscriptionController: Attempting to create local pending PayPal subscription with PayPal ID: {$pendingPaypalIdFromSession}", ['user_id' => $user->id, 'plan_id' => $subscriptionPlan->id]);
                                Subscription::create([
                                    'user_id' => $user->id,
                                    'subscription_plan_id' => $subscriptionPlan->id,
                                    'payment_gateway' => 'paypalgateway', // Ensure this matches the module's key
                                    'gateway_subscription_id' => $pendingPaypalIdFromSession,
                                    'status' => 'pending_approval',
                                    'price_at_purchase' => $subscriptionPlan->price,
                                    'currency_at_purchase' => $subscriptionPlan->currency,
                                ]);
                                Log::info("SubscriptionController: Successfully created local pending PayPal subscription.", ['gateway_subscription_id_stored' => $pendingPaypalIdFromSession]);
                                
                                // Clean up session variables immediately after use
                                session()->forget('paypal_subscription_id_pending');
                                session()->forget('local_plan_id_pending'); // Also clean this if PaypalService sets it
                                
                                return redirect()->away($approvalLink); // Exits early
                            } catch (\Exception $e) {
                                Log::critical("SubscriptionController: EXCEPTION during local pending PayPal Subscription::create()", ['user_id' => $user->id, 'plan_id' => $subscriptionPlan->id, 'attempted_paypal_id' => $pendingPaypalIdFromSession, 'error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
                                return redirect()->route('subscription.checkout', ['subscriptionPlan' => $subscriptionPlan->slug])->with('error', 'Critical error: Failed to record your pending PayPal subscription. Please contact support.');
                            }
                        }
                        $errorMessage = 'Failed to get PayPal approval link.';
                        // $paymentSuccessful remains false
                    } catch (\Exception $e) {
                        Log::error("PayPal Subscription Initiation Error for user {$user->id}, plan {$subscriptionPlan->id}: " . $e->getMessage());
                        $errorMessage = 'An error occurred while initiating PayPal payment: ' . $e->getMessage();
                        // $paymentSuccessful remains false
                    }
                } else {
                    return redirect()->back()->with('error', 'PayPal payment method is currently not available.')->withInput(); // Exits early
                }
            } elseif ($paymentMethodInput === 'paystack_gateway') {
                $paystackModule = Module::find('PaystackGateway');
                if ($paystackModule && $paystackModule->isEnabled() && setting('paystack_enabled', '0') == '1') {
                    try {
                        $paystackService = app(\Modules\PaystackGateway\Services\PaystackService::class);
                        // The service returns an array: ['authorization_url' => ..., 'reference' => ...]
                        // The PaystackSubscriptionController already handles creating the pending subscription
                        // and updating the reference. The main SubscriptionController just needs to initiate.
                        // For consistency, the main SubscriptionController should also create the pending record
                        // if it's not already handled by a module-specific controller that it redirects to.
                        // However, your PaystackSubscriptionController::initializePayment does this.
                        // So, this controller should redirect to the module's initialization route.

                        // Redirect to the Paystack module's initialization route
                        return redirect()->route('subscription.paystack.initialize', ['subscriptionPlan' => $subscriptionPlan->slug]); // Exits early

                    } catch (\Exception $e) {
                        Log::error("Paystack Subscription Initiation Error for user {$user->id}, plan {$subscriptionPlan->id}: " . $e->getMessage());
                        $errorMessage = 'An error occurred while initiating Paystack payment: ' . $e->getMessage();
                    }
                } else {
                    return redirect()->back()->with('error', 'Paystack payment method is currently not available.')->withInput(); // Exits early
                }
            } elseif ($paymentMethodInput === 'stripe_gateway') {
                $stripeModule = Module::find('StripeGateway');
                if ($stripeModule && $stripeModule->isEnabled() && setting('stripe_enabled', '0') == '1') {
                    try {
                        $stripeService = app(\Modules\StripeGateway\Services\StripeService::class);
                        $checkoutSessionUrl = $stripeService->createSubscriptionCheckoutSession($user, $subscriptionPlan);
                        if ($checkoutSessionUrl) {
                            return redirect()->away($checkoutSessionUrl); // Exits early
                        }
                        $errorMessage = 'Failed to create Stripe checkout session.';
                    } catch (\Exception $e) {
                        Log::error("Stripe Subscription Initiation Error for user {$user->id}, plan {$subscriptionPlan->id}: " . $e->getMessage());
                        $errorMessage = 'An error occurred while initiating Stripe payment: ' . $e->getMessage();
                    }
                } else {
                    return redirect()->back()->with('error', 'Stripe payment method is currently not available.')->withInput(); // Exits early
                }
            } elseif ($paymentMethodInput === 'paynow_gateway') { // Add this block
                $paynowModule = Module::find('PaynowGateway');
                if ($paynowModule && $paynowModule->isEnabled() && setting('paynowgateway_enabled', '0') == '1') {
                    try {
                        // Redirect to the Paynow module's initialization route
                        return redirect()->route('subscription.paynow.checkout', ['subscriptionPlan' => $subscriptionPlan->slug]); // Exits early
                    } catch (\Exception $e) {
                        Log::error("Paynow Subscription Initiation Error for user {$user->id}, plan {$subscriptionPlan->id}: " . $e->getMessage());
                        $errorMessage = 'An error occurred while initiating Paynow payment: ' . $e->getMessage();
                    }
                } else {
                    return redirect()->back()->with('error', 'Paynow payment method is currently not available.')->withInput(); // Exits early
                }
            } elseif ($paymentMethodInput === 'custom') {
                if (function_exists('setting') && setting('custom_payment_gateway_enabled', false)) {
                    // Create subscription with a pending status
                    $user->subscriptions()
                        ->whereIn('status', ['active', 'trialing'])
                        ->update([
                            'status' => 'cancelled',
                            'ends_at' => now(),
                            'cancelled_at' => now()
                        ]);

                    Subscription::create([ // Removed $subscription = as it's not used before return
                        'user_id' => $user->id,
                        'subscription_plan_id' => $subscriptionPlan->id,
                        'payment_gateway' => 'custom',
                        'gateway_subscription_id' => 'CUSTOM-' . Str::upper(Str::random(10)),
                        'status' => 'pending_payment', // Or a similar status
                        'starts_at' => now(), // Or null until payment confirmed
                        'ends_at' => null, // Or null until payment confirmed
                    ]);
                    $customInstructions = setting('custom_payment_instructions', 'Please follow the custom payment instructions provided by the administrator.');
                    return redirect()->route('dashboard')->with('info', "Subscription pending. {$customInstructions}"); // Exits early
                } else {
                    $errorMessage = "Custom payment method is not enabled.";
                    // $paymentSuccessful remains false
                }
            } else {
                // Unknown payment method for a PAID plan
                $errorMessage = "The selected payment method '{$paymentMethodInput}' is not recognized for this plan.";
                Log::warning("Unknown payment method '{$paymentMethodInput}' encountered for paid plan ID {$subscriptionPlan->id} by user ID {$user->id}");
                // $paymentSuccessful remains false
            }
        }

        if ($paymentSuccessful) {
            try {
                DB::transaction(function () use ($user, $subscriptionPlan, $finalPaymentMethodForRecord, $gatewayTransactionId, $isNewSubscription) {
                    // Cancel any existing active or trialing subscriptions for the user
                    $user->subscriptions()
                        ->whereIn('status', ['active', 'trialing'])
                        ->update([
                            'status' => 'cancelled',
                            'ends_at' => now(),
                            'cancelled_at' => now()
                        ]);

                    $startDate = now();
                    $endDate = null;
                    if ($subscriptionPlan->interval === 'day') {
                        $endDate = $startDate->copy()->addDays($subscriptionPlan->interval_count);
                    } elseif ($subscriptionPlan->interval === 'week') {
                        $endDate = $startDate->copy()->addWeeks($subscriptionPlan->interval_count);
                    } elseif ($subscriptionPlan->interval === 'month') {
                        $endDate = $startDate->copy()->addMonths($subscriptionPlan->interval_count);
                    } elseif ($subscriptionPlan->interval === 'year') {
                        $endDate = $startDate->copy()->addYears($subscriptionPlan->interval_count);
                    }

                    $subscription = Subscription::create([
                        'user_id' => $user->id,
                        'subscription_plan_id' => $subscriptionPlan->id,
                        'payment_gateway' => $finalPaymentMethodForRecord,
                        'gateway_subscription_id' => $gatewayTransactionId,
                        'status' => $subscriptionPlan->trial_period_days > 0 ? 'trialing' : 'active',
                        'trial_ends_at' => $subscriptionPlan->trial_period_days > 0 ? now()->addDays($subscriptionPlan->trial_period_days) : null,
                        'starts_at' => $startDate,
                        'ends_at' => $endDate,
                        'price_at_purchase' => $subscriptionPlan->price,      // Store price at time of purchase
                        'currency_at_purchase' => $subscriptionPlan->currency, // Store currency at time of purchase
                    ]);

                    if (function_exists('setting') && setting('credits_system_enabled', '0') == '1') {
                        $creditsToAward = $isNewSubscription ? $subscriptionPlan->credits_awarded_on_purchase : $subscriptionPlan->credits_awarded_on_renewal;
                        if ($creditsToAward > 0) {
                            $this->creditService->awardCredits(
                                $user,
                                $creditsToAward,
                                $isNewSubscription ? 'award_subscription_purchase' : 'award_subscription_renewal',
                                "Credits for {$subscriptionPlan->name} subscription",
                                $subscription
                            );
                        }
                    }

                    if (!empty($subscriptionPlan->target_role)) {
                        if (class_exists(Role::class) && Role::where('name', $subscriptionPlan->target_role)->where('guard_name', 'web')->exists()) {
                            $user->syncRoles([$subscriptionPlan->target_role]);
                        } else {
                            Log::warning("Target role '{$subscriptionPlan->target_role}' not found for subscription plan ID {$subscriptionPlan->id}.");
                        }
                    }
                    event(new \App\Events\SubscriptionSuccessfullyProcessed($subscription));
                });
            } catch (\Exception $e) {
                Log::critical("Failed to process subscription within transaction for user {$user->id}, plan {$subscriptionPlan->id}: " . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
                return redirect()->back()->with('error', 'A critical error occurred while finalizing your subscription. Please contact support.')->withInput();
            }

            return redirect()->route('dashboard')->with('success', "Successfully subscribed to {$subscriptionPlan->name}!");
        } else {
            return redirect()->back()->with('error', $errorMessage)->withInput();
        }
    }
}