<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\User; // For notifying admins
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
// use App\Notifications\NewPublicTicketNotification; // Create this notification
// use App\Notifications\PublicTicketConfirmationNotification; // Create this notification

class ContactPageController extends Controller
{
    public function showContactForm()
    {
        $contactPhone = setting('contact_phone');
        $contactEmail = setting('contact_email');
        $whatsappNumber = setting('whatsapp_number');
        $whatsappMessage = setting('whatsapp_default_message', 'Hello, I have a question.');

        return view('frontend.contact', compact('contactPhone', 'contactEmail', 'whatsappNumber', 'whatsappMessage'));
    }

    public function submitContactForm(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
            'priority' => 'required|in:low,medium,high',
        ]);

        $userId = Auth::check() ? Auth::id() : null;
        $guestName = !$userId ? $request->name : null;
        $guestEmail = !$userId ? $request->email : null;

        try {
            $ticket = Ticket::create([
                'user_id' => $userId,
                'guest_name' => $guestName, // Add these to your Ticket model's $fillable if you add these columns
                'guest_email' => $guestEmail, // Add these to your Ticket model's $fillable if you add these columns
                'subject' => $request->subject,
                'message' => $request->message,
                'priority' => $request->priority,
                'status' => 'open',
                'last_reply_at' => now(),
            ]);

            // Notify admins
            // $adminUsers = User::where('is_admin', true)->get();
            // foreach ($adminUsers as $admin) {
            //     $admin->notify(new NewPublicTicketNotification($ticket));
            // }

            // Send confirmation to the submitter
            // Mail::to($request->email)->send(new PublicTicketConfirmationMail($ticket));

            return redirect()->route('frontend.contact.form')->with('success', 'Your message has been sent successfully! We will get back to you shortly.');

        } catch (\Exception $e) {
            Log::error('Error submitting public contact form: ' . $e->getMessage(), [
                'request_data' => $request->except('password', '_token'), // Don't log sensitive data
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->back()->withInput()->with('error', 'Sorry, there was an issue submitting your message. Please try again later.');
        }
    }
}
