<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\User; // Assuming your User model is in App\Models
use App\Models\Setting;
use Illuminate\Support\Str; // Import Str
class BoilerplateActivationController extends Controller
{
    // IMPORTANT: Define the slug for your boilerplate product as it's known by AppManager
    // This should match a 'ManagedScript' slug in your AppManager module.
    private string $productSlug;

    // The full URL to your AppManager module's API endpoints
    private string $appManagerApiBaseUrl;

    public function __construct()
    {
        // These should ideally come from your boilerplate's .env or config files
        $this->productSlug = config('boilerplate.product_slug', 'my-boilerplate-slug'); // Read from config/boilerplate.php
        $this->appManagerApiBaseUrl = rtrim(config('boilerplate.appmanager_url', url('/')), '/') . '/api/app-manager'; // Read from config/boilerplate.php

        // Ensure product slug is set
        if (empty($this->productSlug) || $this->productSlug === 'my-boilerplate-slug') {
            Log::warning('BoilerplateActivationController: BOILERPLATE_PRODUCT_SLUG is not configured or using default. Please set it in your .env file.');
            // Optionally, throw an exception or prevent access if not configured, or ensure a sensible default in config/boilerplate.php
        }
    }

    public function showActivationForm()
    {
        // Use a file-based flag to check for completed installation
        // This avoids DB access before it's configured.
        $installFlagPath = storage_path('app/installed.flag');

        if (File::exists($installFlagPath)) {
            // Attempt to read DB setting only if flag file exists, for a more graceful redirect message
            $dbInstallComplete = false;
            try {
                $dbInstallComplete = Setting::getValue('boilerplate_install_complete', 'false') === 'true';
            } catch (\Exception $e) {
                // DB might not be accessible, but install flag exists, so treat as installed.
            }
            return redirect()->route(config('boilerplate.post_install_redirect_route', 'login')) // Use a configurable redirect
                             ->with('info', 'Application is already installed and activated.');
        }
        // Create this view in your boilerplate: resources/views/admin/boilerplate/activation_form.blade.php
        return view('admin.boilerplate.activation_form');
    }

    public function showSetupForm(Request $request)
    {
        // Ensure this step is only accessed after successful license pre-check
        if (Session::get('installation_step') !== 'setup_db_admin' || !Session::has('activation_data')) {
            return redirect()->route('install.activate.form')->with('error', 'Please complete the license activation step first.');
        }

        $activationData = Session::get('activation_data');
        // Pre-fill domain if available from previous step
        $domain = $activationData['domain'] ?? request()->getSchemeAndHttpHost();

        // Create this view: resources/views/admin/boilerplate/setup_form.blade.php
        return view('admin.boilerplate.setup_form', compact('domain'));
    }

    public function processActivation(Request $request)
    {
        $request->validate([
            'license_key' => 'required_without:purchase_code|nullable|string|max:255',
            'purchase_code' => 'required_without:license_key|nullable|string|max:255',
            'domain' => 'required|string|max:255|url', // Domain for which license is being activated
        ]);

        if (empty($request->input('license_key')) && empty($request->input('purchase_code'))) {
            return back()->withInput()->with('error', 'Either a License Key or a Purchase Code is required.');
        }

        $payload = [
            'product_slug' => $this->productSlug,
            // Ensure domain is consistently stored/sent in lowercase
            'domain' => strtolower(trim($request->input('domain'))),
        ];

        if ($request->filled('license_key')) {
            $payload['license_key'] = $request->input('license_key');
        } elseif ($request->filled('purchase_code')) {
            $payload['purchase_code'] = $request->input('purchase_code');
        }

        try {
            $activationApiUrl = $this->appManagerApiBaseUrl . '/activate';
            $response = Http::timeout(30)->post($activationApiUrl, $payload);
            $responseData = $response->json();

            if ($response->successful() && isset($responseData['status']) && $responseData['status'] === 'success') {
                // License is valid with AppManager. Store data temporarily in session and proceed to next step.
                $usedLicenseIdentifier = $request->filled('license_key') ? $request->input('license_key') : $request->input('purchase_code');
                $activatedDomain = strtolower(trim($request->input('domain')));

                Session::put('installation_step', 'setup_db_admin');
                Session::put('activation_data', [
                    'license_identifier' => $usedLicenseIdentifier,
                    'activated_domain' => $activatedDomain,
                    'appmanager_response' => $responseData, // Store the whole response for token, version etc.
                    // Store the specific license key from AppManager if it was a purchase code activation
                    'appmanager_license_key' => $responseData['license_info']['license_key'] ?? ($request->filled('license_key') ? $request->input('license_key') : null),
                ]);

                Log::info('Boilerplate: License pre-check successful. Proceeding to DB/Admin setup.', ['domain' => $activatedDomain, 'product_slug' => $this->productSlug]);
                return redirect()->route('install.setup.form');
            } else {
                $errorMessage = $responseData['message'] ?? 'Activation failed. Please check your details and try again.';
                if (isset($responseData['errors'])) {
                    Log::error('Boilerplate activation API validation failed.', ['errors' => $responseData['errors'], 'payload' => $payload]);
                    $errorMessage .= " Details: " . collect($responseData['errors'])->flatten()->implode(' ');
                } else {
                    Log::error('Boilerplate activation failed via AppManager API.', ['status_code' => $response->status(), 'response_body' => $responseData, 'payload' => $payload]);
                }
                return back()->withInput()->with('error', $errorMessage);
            }
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::critical('Boilerplate activation: Could not connect to AppManager API.', ['error' => $e->getMessage(), 'url' => $activationApiUrl]);
            return back()->withInput()->with('error', 'Could not connect to the activation server. Please try again later.');
        } catch (\Exception $e) {
            Log::critical('Boilerplate activation: An unexpected error occurred.', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return back()->withInput()->with('error', 'An unexpected error occurred during activation. Please check the logs.');
        }
    }

    public function validateCodeAjax(Request $request)
    {
        $request->validate([
            'license_key' => 'nullable|string|max:255',
            'purchase_code' => 'nullable|string|max:255',
            'domain' => 'required|string|max:255|url',
            'code_type' => 'required|in:license_key,purchase_code', // To know which field triggered
        ]);

        if (empty($request->input('license_key')) && empty($request->input('purchase_code'))) {
            return response()->json(['valid' => false, 'message' => 'A License Key or Purchase Code is required.'], 422);
        }

        $payload = [
            'product_slug' => $this->productSlug,
            'domain' => strtolower(trim($request->input('domain'))),
        ];

        $codeToValidate = '';
        if ($request->input('code_type') === 'license_key' && $request->filled('license_key')) {
            $payload['license_key'] = $request->input('license_key');
            $codeToValidate = $request->input('license_key');
        } elseif ($request->input('code_type') === 'purchase_code' && $request->filled('purchase_code')) {
            $payload['purchase_code'] = $request->input('purchase_code');
            $codeToValidate = $request->input('purchase_code');
        }

        if (empty($codeToValidate)) {
             return response()->json(['valid' => false, 'message' => 'No code provided for validation.'], 422);
        }

        try {
            $validationApiUrl = $this->appManagerApiBaseUrl . '/activate'; // Using the same activation endpoint for validation
            Log::debug('Validating code via AppManager API', ['url' => $validationApiUrl, 'payload' => $payload]);
            $response = Http::timeout(15)->post($validationApiUrl, $payload);
            $responseData = $response->json();
            Log::debug('AppManager API validation response', ['status' => $response->status(), 'body' => $responseData]);

            if ($response->successful() && isset($responseData['status']) && $responseData['status'] === 'success') {
                $licenseInfo = $responseData['license_info'] ?? [];
                $licenseStatus = $licenseInfo['status'] ?? 'unknown';
                // AppManager should ideally return a clear 'type'. This is an attempt to infer.
                $licenseType = $licenseInfo['type'] ?? ($request->input('code_type') === 'license_key' ? 'License Key' : 'Purchase Code');

                return response()->json([
                    'valid' => true,
                    'status' => $licenseStatus,
                    'type' => $licenseType,
                    'message' => $responseData['message'] ?? 'Code is valid.',
                    'details' => $licenseInfo
                ]);
            } else {
                $errorMessage = $responseData['message'] ?? 'Validation failed.';
                if (isset($responseData['errors'])) {
                    $errorMessage .= " Details: " . collect($responseData['errors'])->flatten()->implode(' ');
                }
                return response()->json(['valid' => false, 'message' => $errorMessage, 'status' => $responseData['license_info']['status'] ?? ($responseData['status'] ?? 'invalid')], $response->status() >= 400 && $response->status() < 500 ? $response->status() : 400);
            }
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Boilerplate code validation: Could not connect to AppManager API.', ['error' => $e->getMessage(), 'url' => $validationApiUrl]);
            return response()->json(['valid' => false, 'message' => 'Could not connect to the validation server. Please try again later.'], 503);
        } catch (\Exception $e) {
            Log::critical('Boilerplate code validation: An unexpected error occurred.', ['error' => $e->getMessage()]);
            return response()->json(['valid' => false, 'message' => 'An unexpected error occurred during validation. Please check the logs.'], 500);
        }
    }

    public function processSetup(Request $request)
    {
        // TEMPORARY DEBUGGING - REMOVE FOR PRODUCTION
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
        ini_set('display_startup_errors', 1); // Important for errors during PHP startup
        ob_implicit_flush(true); // Try to force output
        // END TEMPORARY DEBUGGING

        // Attempt to increase execution time for this potentially long process
        @ini_set('max_execution_time', 300); // 300 seconds = 5 minutes
        @set_time_limit(300);

        // Ensure this step is only accessed after successful license pre-check
        if (Session::get('installation_step') !== 'setup_db_admin' || !Session::has('activation_data')) {
            return redirect()->route('install.activate.form')->with('error', 'Invalid setup sequence. Please start over.');
        }

        $request->validate([
            'db_host' => 'required|string|max:255',
            'db_port' => 'required|integer|min:1|max:65535',
            'db_database' => 'required|string|max:255',
            'db_username' => 'required|string|max:255',
            'db_password' => 'nullable|string|max:255', // Password can be empty for some local setups
            'admin_name' => 'required|string|max:255',
            'admin_email' => 'required|string|email|max:255',
            'admin_password' => 'required|string|min:8|confirmed',
        ]);

        // 1. Test Database Connection & Update .env
        $dbConfig = [
            'DB_HOST' => $request->input('db_host'),
            'DB_PORT' => $request->input('db_port'),
            'DB_DATABASE' => $request->input('db_database'),
            'DB_USERNAME' => $request->input('db_username'),
            'DB_PASSWORD' => $request->input('db_password') ?? '',
        ];

        if (!$this->testAndSetDatabaseConnection($dbConfig)) {
            return back()->withInput()->with('error', 'Could not connect to the database with the provided credentials. Please check and try again.');
        }

        // 2. Ensure APP_KEY is set in .env
        try {
            // Reload config to pick up .env changes from testAndSetDatabaseConnection
            Artisan::call('config:clear'); // Clear first to ensure fresh load
            // Check if APP_KEY is empty or still the placeholder from .env.example
            $currentAppKey = config('app.key'); // Load it after config:clear
            if (empty($currentAppKey) || str_starts_with($currentAppKey, 'base64:someDefaultPlaceholder') || $currentAppKey === 'base64:'.base64_encode(str_repeat("\0", 32))) { // Check against common placeholders
                Log::info('Boilerplate: APP_KEY is missing or default. Generating new key.');
                Artisan::call('key:generate', ['--force' => true]);
                Artisan::call('config:clear'); // Clear again after key generation
            }
        } catch (\Exception $e) {
            Log::error('Boilerplate: Failed to generate APP_KEY.', ['error' => $e->getMessage()]);
            // This is not necessarily fatal for installation but should be noted.
            Session::flash('warning', 'Failed to automatically generate application key. Please run `php artisan key:generate` manually after installation.');
        }

        // 2a. Create cache table if using database cache (run before main migrations)
        // We will switch to database cache later, so ensure the table exists.
        try {
            Log::info('Boilerplate: Attempting to run cache:table command.');
            Artisan::call('cache:table');
            $cacheTableOutput = trim(Artisan::output());
            if (Str::contains($cacheTableOutput, 'Migration already exists.', true)) {
                Log::info('Boilerplate: cache:table command reported migration already exists (which is fine).', ['output' => $cacheTableOutput]);
            } elseif (Str::contains($cacheTableOutput, ['Error', 'Failed', 'failed'], true) && !empty($cacheTableOutput)) {
                Log::error('Boilerplate: cache:table command indicated an error (other than migration exists).', ['output' => $cacheTableOutput]);
            } else {
                Log::info('Boilerplate: cache:table command executed.', ['output' => $cacheTableOutput]);
            }
        } catch (\Exception $e) {
            Log::warning('Boilerplate: cache:table command failed or table already exists.', ['error' => $e->getMessage()]);
            // Continue, as migrate should create it if it doesn't exist and migration file is present
        }

        // 2b. Run All Migrations (including settings table, users, roles, cache table etc.)
        try {
            Artisan::call('migrate', ['--force' => true]); // --force for non-interactive
            $migrationOutput = Artisan::output();
            if (Str::contains($migrationOutput, ['Error', 'Failed', 'failed'])) {
                Log::error('Boilerplate: Database migrations indicated an error.', ['output' => $migrationOutput]);
                return back()->withInput()->with('error', 'Database migration encountered an issue. Please check logs.');
            }
            Log::info('Boilerplate: Database migrations run successfully during installation. Output: ' . $migrationOutput);
        } catch (\Exception $e) {
            Log::critical('Boilerplate: Database migration failed during installation.', ['error' => $e->getMessage()]);
            return back()->withInput()->with('error', 'Database migration failed: ' . $e->getMessage());
        }

        // 2c. Seed Default Application Settings
        try {
            Log::info('Boilerplate: Seeding default application settings.');
            Artisan::call('db:seed', ['--class' => 'SettingsTableSeeder', '--force' => true]);
            $seederOutput = trim(Artisan::output());
            Log::info('Boilerplate: SettingsTableSeeder executed.', ['output' => $seederOutput]);
            // Check seeder output for common failure indicators if Artisan call itself doesn't throw exception
            if (Str::contains($seederOutput, ['Error', 'Failed', 'failed', 'Exception'], true) || !Str::contains($seederOutput, 'seeded successfully', true)) {
                Log::error('Boilerplate: SettingsTableSeeder may have encountered an issue. Review output.', ['output' => $seederOutput]);
                // Decide if this is fatal. For now, we'll log and continue, but some settings might be missing.
                Session::flash('warning', 'Could not seed all default application settings. Some configurations might need manual setup. Please check logs.');
            }
        } catch (\Exception $e) {
            Log::error('Boilerplate: Failed to seed default application settings.', ['error' => $e->getMessage()]);
            return back()->withInput()->with('error', 'Failed to seed default application settings: ' . $e->getMessage());
        }

        // 3. Switch Cache Driver to Database (now that migrations, including cache table, should be done)
        Log::info('Boilerplate: Attempting to switch cache driver to database in .env.');
        if ($this->updateEnvFile(['CACHE_STORE' => 'database'])) {
            Artisan::call('config:clear');
            Artisan::call('cache:clear'); 
            Log::info('Boilerplate: Cache driver switched to database and caches cleared.');
        } else {
            Log::error('Boilerplate: Failed to update .env to switch cache driver to database.');
            // Not necessarily fatal for this step, but settings won't be DB cached.
            // The Setting model's conditional tagging will still prevent errors with 'file' cache.
            Session::flash('warning', 'Could not switch cache driver to database automatically. Settings caching might use file system.');
        }

        // Crucial: Re-establish the default database connection with the new .env settings
        // This ensures subsequent DB operations (like User::create, Setting::setValue) use the correct config.
        DB::purge(config('database.default'));
        DB::reconnect(config('database.default'));
        Log::info('Boilerplate: Database connection purged and reconnected after .env updates.');



        // 4. Create Admin User
        try {
            $adminUser = User::create([
                'name' => $request->input('admin_name'),
                'email' => $request->input('admin_email'),
                'password' => Hash::make($request->input('admin_password')),
                'email_verified_at' => now(), // Auto-verify admin
            ]);
            Log::info('Boilerplate: Admin user CREATED in database.', ['user_id' => $adminUser->id, 'email' => $adminUser->email]);
            // Assign 'super_admin' or 'admin' role. Ensure these roles exist (e.g., seeded by Spatie permissions package)
            if (method_exists($adminUser, 'assignRole')) {
                 // Check if roles exist before assigning
                if (\Spatie\Permission\Models\Role::where('name', 'super_admin')->exists()) {
                    $adminUser->assignRole('super_admin');
                } elseif (\Spatie\Permission\Models\Role::where('name', 'admin')->exists()) {
                    $adminUser->assignRole('admin');
                } else {
                    Log::warning('Boilerplate: Default admin roles (super_admin, admin) not found. Admin user created without a role.');
                }
            }
            Log::info('Boilerplate: Admin user roles assigned (if applicable).', ['email' => $adminUser->email]);
        } catch (\Exception $e) {
            Log::critical('Boilerplate: Admin user creation failed.', ['error' => $e->getMessage()]);
            return back()->withInput()->with('error', 'Failed to create admin user: ' . $e->getMessage());
        }

        // 5. Retrieve activation data from session and store permanently (Now using database cache if switch was successful)
        $activationSessionData = Session::get('activation_data');
        if (!$activationSessionData) {
            Log::critical('Boilerplate: Activation data missing from session during setup.');
            return redirect()->route('install.activate.form')->with('error', 'Activation data lost. Please start over.');
        }

        // Pass the necessary parts of activationSessionData directly
        if (!$this->storeActivationDataPermanently(
            $activationSessionData['license_identifier'],
            $activationSessionData['activated_domain'],
            $activationSessionData // Pass the whole array for internal use
        )) {
            Log::critical('Boilerplate: Failed to store activation data permanently in settings.');
            return back()->withInput()->with('error', 'Failed to save activation settings. Please check logs.');
        }
        // 6. Download Core Files (excluding config/app.php)
        $appManagerResponseFromSession = $activationSessionData['appmanager_response'] ?? [];
        $activationToken = $appManagerResponseFromSession['activation_token'] ?? null;
        $productVersion = $appManagerResponseFromSession['product_info']['version'] ?? 'latest';
        
        if ($activationToken) {
            $filesDownloaded = $this->downloadCoreFiles($activationToken, $productVersion);
            if (!$filesDownloaded) {
                Log::error('Boilerplate: Critical core file download failed. Installation cannot be marked as complete.');
                // Optionally, clean up previous steps or .env changes if possible
                return back()->withInput()->with('error', 'A critical error occurred while downloading necessary application files. Please check logs and try again.');
            }
        } else {
            Log::error('Boilerplate: Activation token missing from session during setup, cannot download core files.');
            return back()->withInput()->with('error', 'Activation session data is missing. Unable to download core files. Please start over.');
            // This shouldn't happen if the flow is correct, but good to log.
        }

        // 7. Mark Installation Complete (writes to DB settings and creates flag file)
        if (!$this->markInstallationComplete()) {
            Log::critical('Boilerplate: CRITICAL - Failed to mark installation as complete. Setup aborted before final steps.');
            // At this point, .env is updated, migrations run. User might need to manually fix settings or flag.
            return back()->withInput()->with('error', 'Failed to finalize installation. Please check system logs and ensure storage/app is writable.');
        }

        // 8. Clean up session
        Session::forget(['installation_step', 'activation_data']);

        // 9. Log in the new admin user
        Auth::login($adminUser);

        Log::info('Boilerplate installation completed successfully.');
        Session::flash('installation_complete', true); // Add a specific session flash for the login page to check
        return redirect()->route('login') // Redirect to login page
                         ->with('success', 'Installation complete! Please log in with your new admin credentials to continue.');
    }

    protected function storeActivationDataPermanently(string $userInputtedIdentifier, string $activatedDomain, array $activationSessionData): bool
    {
        // $activationSessionData is now passed as an argument
        if (!$activationSessionData || !isset($activationSessionData['appmanager_response'])) {
            Log::error('Boilerplate: AppManager response data missing from session when trying to store settings.');
            return false; // Indicate failure
        }

        try {
            $activationResponseData = $activationSessionData['appmanager_response'];

            $token = $activationResponseData['activation_token'] ?? null;
            $productVersion = $activationResponseData['product_info']['version'] ?? null;
            $licenseStatus = $activationResponseData['license_info']['status'] ?? null;
            $finalLicenseKeyToStore = $activationSessionData['appmanager_license_key'] ?? $userInputtedIdentifier;

            if ($token) {
                // Pass name, type, group for consistency with seeder and Setting model expectations
                Setting::setValue('boilerplate_activation_token', $token, 'Boilerplate Activation Token', 'text', 'Installation');
                Setting::setValue('boilerplate_license_key', $finalLicenseKeyToStore, 'Boilerplate License Key', 'text', 'Installation');
                Setting::setValue('boilerplate_activated_version', $productVersion, 'Boilerplate Activated Version', 'text', 'Installation');
                Setting::setValue('boilerplate_license_status', $licenseStatus, 'Boilerplate License Status', 'text', 'Installation');
                Setting::setValue('boilerplate_activated_domain', $activatedDomain, 'Boilerplate Activated Domain', 'text', 'Installation');
                Log::info('Boilerplate: Activation data stored in settings table.');
                return true; // Indicate success
            } else {
                Log::error('Boilerplate: Activation token was null, cannot store activation settings.');
                return false; // Indicate failure
            }
        } catch (\Exception $e) {
            Log::critical('Boilerplate: Exception while storing activation data in settings.', ['error' => $e->getMessage()]);
            return false; // Indicate failure
        }
    }

    protected function downloadCoreFiles(string $activationToken, string $productVersion): bool
    {
        $filesToDownload = [
            ['file_identifier' => 'app/Providers/AuthServiceProvider.php', 'local_save_path' => app_path('Providers/AuthServiceProvider.php')],
            ['file_identifier' => 'app/Helpers/UserAccessHelper.php',        'local_save_path' => app_path('Helpers/UserAccessHelper.php')],
        // REMOVED: ['file_identifier' => 'config/app.php', 'local_save_path' => config_path('app.php')],
        ];

        $downloadApiUrl = $this->appManagerApiBaseUrl . '/download-file';
        $allSuccessful = true;

        foreach ($filesToDownload as $fileInfo) {
            try {
                $payload = [
                    'activation_token' => $activationToken,
                    'product_slug'     => $this->productSlug,
                    'file_identifier'  => $fileInfo['file_identifier'],
                    'version'          => $productVersion,
                ];

                Log::info('Boilerplate: Attempting to download core file.', $payload);
                $response = Http::timeout(60)->post($downloadApiUrl, $payload);

                if ($response->successful() && strpos($response->header('Content-Type'), 'application/json') === false) {
                    File::ensureDirectoryExists(dirname($fileInfo['local_save_path']));
                    File::put($fileInfo['local_save_path'], $response->body());
                    Log::info('Boilerplate: Successfully downloaded and saved core file.', ['identifier' => $fileInfo['file_identifier'], 'save_path' => $fileInfo['local_save_path']]);

                    $serverHash = $response->header('X-File-Hash');
                    if ($serverHash) {
                        $localHash = hash_file('sha256', $fileInfo['local_save_path']);
                        if ($localHash !== $serverHash) {
                            Log::warning('Boilerplate: Downloaded file hash mismatch.', ['identifier' => $fileInfo['file_identifier'], 'server_hash' => $serverHash, 'local_hash' => $localHash]);
                            // Consider deleting the file or marking activation as incomplete
                        }
                    }
                } else {
                    $responseData = $response->json() ?? ['message' => 'Unknown error during file download. Response not JSON.'];
                    $errorMessage = $responseData['message'] ?? "Failed to download file: {$fileInfo['file_identifier']}. Status: " . $response->status();
                    Log::error('Boilerplate: Failed to download core file from AppManager.', ['identifier' => $fileInfo['file_identifier'], 'status_code' => $response->status(), 'response_body' => $responseData, 'payload' => $payload]);
                    Session::flash('error_file_' . Str::slug($fileInfo['file_identifier']), $errorMessage);
                    $allSuccessful = false;
                }
            } catch (\Exception $e) {
                Log::critical('Boilerplate: Exception while downloading core file.', ['error' => $e->getMessage(), 'file' => $fileInfo['file_identifier']]);
                Session::flash('error_file_' . Str::slug($fileInfo['file_identifier']), "An unexpected error occurred while downloading {$fileInfo['file_identifier']}.");
                $allSuccessful = false;
            }
        }
        return $allSuccessful;
    }

    // Helper for .env (simplified, use a robust package for production if you choose this method)
    protected function updateEnvFile(array $values)
    {
        $envPath = base_path('.env');
        $exampleEnvPath = base_path('.env.example');

        if (!File::exists($envPath)) {
            Log::info('Boilerplate: .env file not found. Attempting to create from .env.example.');
            if (File::exists($exampleEnvPath)) {
                try {
                    File::copy($exampleEnvPath, $envPath);
                    Log::info('Boilerplate: .env file created successfully from .env.example.');
                } catch (\Exception $e) {
                    Log::critical('Boilerplate: Failed to copy .env.example to .env.', ['error' => $e->getMessage()]);
                    return false; // Indicate failure
                }
            } else {
                Log::critical('Boilerplate: .env.example file not found. Cannot create .env file.');
                return false; // Indicate failure
            }
        }

        $content = File::get($envPath);

        foreach ($values as $key => $value) {
            $escapedValue = (is_string($value) && (str_contains($value, ' ') || str_contains($value, '#'))) ? "\"{$value}\"" : $value;
            $keyToReplace = strtoupper($key);

            if (str_contains($content, $keyToReplace . '=')) {
                $content = preg_replace(
                    "/^{$keyToReplace}=.*/m",
                    "{$keyToReplace}={$escapedValue}",
                    $content
                );
            } else {
                $content .= "\n{$keyToReplace}={$escapedValue}";
            }
        }
        File::put($envPath, $content);
        return true; // Indicate success
    }

    protected function markInstallationComplete(): bool
    {
        try {
            // Attempt to set the database flag
            Setting::setValue('boilerplate_install_complete', 'true', 'Boilerplate Installation Complete', 'boolean', 'Installation');
            Log::info('Boilerplate: Database setting "boilerplate_install_complete" set to true.');

            // Create the file-based flag
            $installFlagPath = storage_path('app/installed.flag');
            if (!File::isDirectory(dirname($installFlagPath))) {
                File::makeDirectory(dirname($installFlagPath), 0755, true, true);
            }
            File::put($installFlagPath, time());
            Log::info('Boilerplate: Installation flag file created.', ['path' => $installFlagPath]);

            // Logic to prompt for deletion of an 'install' directory
            $installDirPath = public_path('install_init_temp'); // Example path, adjust if needed
            if (File::isDirectory($installDirPath)) {
                Session::flash('warning_install_delete', 'For security, please manually delete the installation directory: <strong>' . $installDirPath . '</strong> (if it exists).');
            }
            return true; // Indicate success
        } catch (\Exception $e) {
            Log::critical('Boilerplate: Failed to mark installation as complete.', [
                'error' => $e->getMessage(),
                'trace' => Str::limit($e->getTraceAsString(), 1000) // Limit trace length
            ]);
            return false; // Indicate failure
        }
    }

    protected function testAndSetDatabaseConnection(array $dbConfig): bool
    {
        // Temporarily set the DB config for testing
        config([
            'database.connections.install_test' => [
                'driver' => 'mysql', // Assuming MySQL, adjust if needed
                'host' => $dbConfig['DB_HOST'],
                'port' => $dbConfig['DB_PORT'],
                'database' => $dbConfig['DB_DATABASE'],
                'username' => $dbConfig['DB_USERNAME'],
                'password' => $dbConfig['DB_PASSWORD'],
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
                'prefix' => '',
                'strict' => true,
                'engine' => null,
            ],
        ]);

        try {
            DB::connection('install_test')->getPdo();
            Log::info('Boilerplate: Database connection test successful.');
            // If successful, update the .env file.
            // updateEnvFile now returns bool, so we check its success.
            if (!$this->updateEnvFile($dbConfig)) {
                return false; // Propagate failure if .env handling failed
            }
            return true;
        } catch (\Exception $e) {
            Log::error('Boilerplate: Database connection test failed.', ['error' => $e->getMessage()]);
            return false;
        }
    }
}