<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Laravel\Socialite\Facades\Socialite;
use Illuminate\Support\Str;
use Exception; // For general exception catching
use InvalidArgumentException; // For Socialite driver issues
use Laravel\Socialite\Two\InvalidStateException; // For OAuth state issues

class SocialLoginController extends Controller
{
    /**
     * Redirect the user to the provider's authentication page.
     *
     * @param  string $provider
     * @return \Illuminate\Http\RedirectResponse|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function redirectToProvider(string $provider)
    {
        if (! $this->isProviderAllowedAndConfigured($provider)) {
            return redirect()->route('login')->with('error', ucfirst($provider) . ' login is not enabled or properly configured.');
        }

        try {
            return Socialite::driver($provider)->redirect();
        } catch (InvalidArgumentException $e) {
            report($e); // Log the actual error for debugging
            return redirect()->route('login')->with('error', 'Configuration error for ' . ucfirst($provider) . ' login. Please contact support.');
        } catch (Exception $e) {
            report($e);
            return redirect()->route('login')->with('error', 'An unexpected error occurred while trying to connect with ' . ucfirst($provider) . '. Please try again.');
        }
    }

    /**
     * Obtain the user information from the provider.
     *
     * @param  string $provider
     * @return \Illuminate\Http\RedirectResponse
     */
    public function handleProviderCallback(string $provider)
    {
        if (! $this->isProviderAllowedAndConfigured($provider)) {
            return redirect()->route('login')->with('error', ucfirst($provider) . ' login is not enabled or properly configured.');
        }

        try {
            $socialUser = Socialite::driver($provider)->user();
        } catch (InvalidStateException $e) {
            return redirect()->route('login')->with('error', 'Invalid session state. Please try logging in again.');
        } catch (Exception $e) {
            report($e);
            return redirect()->route('login')->with('error', 'Unable to login using ' . ucfirst($provider) . '. Please try again.');
        }

        // Ensure we have an email, as it's usually essential
        if (empty($socialUser->getEmail())) {
            return redirect()->route('login')->with('error', 'Could not retrieve email address from ' . ucfirst($provider) . '. Please ensure your ' . ucfirst($provider) . ' account has a primary email address and it is shared.');
        }

        $user = User::where('provider_name', strtolower($provider))
            ->where('provider_id', $socialUser->getId())
            ->first();

        if (!$user) {
            // User not found by provider_id, try by email
            $user = User::where('email', $socialUser->getEmail())->first();

            if ($user) {
                // User with this email already exists. Link the social account.
                $user->provider_name = strtolower($provider);
                $user->provider_id = $socialUser->getId();
                // Optionally update avatar or other details if you store them
                // $user->avatar_url = $socialUser->getAvatar();
                $user->save();
            } else {
                // No user found by provider_id or email, create a new one
                $user = User::create([
                    'name' => $socialUser->getName() ?? $socialUser->getNickname() ?? 'User_' . Str::random(5),
                    'email' => $socialUser->getEmail(),
                    'provider_name' => strtolower($provider),
                    'provider_id' => $socialUser->getId(),
                    'email_verified_at' => now(), // Assume email from social provider is verified
                    'password' => Hash::make(Str::random(24)), // Set a random secure password
                    // 'avatar_url' => $socialUser->getAvatar(), // if you store avatar
                ]);
            }
        }

        Auth::login($user, true); // Log in the user with "remember me"

        return redirect()->intended(config('fortify.home', '/dashboard'));
    }

    /**
     * Check if the social login provider is allowed and configured.
     *
     * @param string $provider
     * @return bool
     */
    protected function isProviderAllowedAndConfigured(string $provider): bool
    {
        $providerKey = strtolower($provider);
        $allowedProviders = ['google', 'facebook']; // Define your allowed providers

        if (!in_array($providerKey, $allowedProviders)) {
            return false;
        }

        // Check settings if the provider is enabled
        // Ensure your setting() helper is available and works correctly
        if (!function_exists('setting') || setting("{$providerKey}_login_enabled", '0') !== '1') {
            return false;
        }

        // Check if essential credentials are set in config/services.php
        // (which should pull from settings or .env)
        $clientIdKey = ($providerKey === 'facebook') ? 'facebook_app_id' : "{$providerKey}_client_id";
        $clientSecretKey = ($providerKey === 'facebook') ? 'facebook_app_secret' : "{$providerKey}_client_secret";

        if (empty(config("services.{$providerKey}.client_id")) || empty(config("services.{$providerKey}.client_secret"))) {
            // Log this issue for the admin, as it's a configuration problem
            report("Social login for {$providerKey} is enabled in settings, but client_id or client_secret is missing in config/services.php or not pulled correctly from settings/env.");
            return false;
        }

        return true;
    }
}
