<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\TicketReply;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Notifications\TicketStatusUpdatedNotification; // We'll create this later
use App\Notifications\NewTicketReplyNotification; // We'll create this later

class TicketController extends Controller
{
    public function index(Request $request)
    {
        $query = Ticket::with('user')->orderBy('last_reply_at', 'desc')->orderBy('created_at', 'desc');

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $tickets = $query->paginate(15);
        $statuses = ['open', 'pending_user_reply', 'pending_admin_reply', 'closed']; // Example statuses

        return view('admin.tickets.index', compact('tickets', 'statuses'));
    }

    public function show(Ticket $ticket)
    {
        $admins = User::where('is_admin', true)->get(); // For assigning tickets
        return view('admin.tickets.show', compact('ticket', 'admins'));
    }

    public function reply(Request $request, Ticket $ticket)
    {
        if ($ticket->status === 'closed') {
            return back()->with('error', 'This ticket is closed and cannot be replied to.');
        }

        $request->validate([
            'message' => 'required|string|max:5000',
        ]);

        TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(), // Admin's ID
            'message' => $request->message,
        ]);

        $ticket->update([
            'status' => 'pending_user_reply',
            'last_reply_at' => now(),
        ]);

        // Notify the user who created the ticket
        // $ticket->user->notify(new NewTicketReplyNotification($ticket, Auth::user()));

        return redirect()->route('admin.tickets.show', $ticket)->with('success', 'Reply submitted successfully.');
    }

    public function updateStatus(Request $request, Ticket $ticket)
    {
        $request->validate([
            'status' => 'required|in:open,pending_user_reply,pending_admin_reply,closed',
        ]);

        $oldStatus = $ticket->status;
        $ticket->update(['status' => $request->status]);

        // Notify user if status changed significantly
        // if ($oldStatus !== $request->status) {
        //    $ticket->user->notify(new TicketStatusUpdatedNotification($ticket));
        // }

        return redirect()->route('admin.tickets.show', $ticket)->with('success', 'Ticket status updated.');
    }

    public function assign(Request $request, Ticket $ticket)
    {
        $request->validate([
            'assigned_to' => 'nullable|exists:users,id',
        ]);

        $assignedAdminId = $request->input('assigned_to');
        $ticket->update(['assigned_to' => $assignedAdminId]);

        // Notify the newly assigned admin (optional)
        // if ($assignedAdminId && $ticket->assignedAdmin) {
        //     $ticket->assignedAdmin->notify(new TicketAssignedNotification($ticket));
        // }

        return redirect()->route('admin.tickets.show', $ticket)->with('success', 'Ticket assignment updated.');
    }
}