<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use Nwidart\Modules\Facades\Module;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB; // Import DB Facade
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role; // Import Spatie Role model

class SubscriptionPlanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (function_exists('setting') && setting('subscriptions_enabled', '0') != '1') {
            return redirect()->route('admin.dashboard')->with('info', 'Subscription feature is currently disabled.');
        }
        $plans = SubscriptionPlan::orderBy('name')->paginate(15);
        return view('admin.subscription_plans.index', compact('plans'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (function_exists('setting') && setting('subscriptions_enabled', '0') != '1') {
            return redirect()->route('admin.dashboard')->with('info', 'Subscription feature is currently disabled.');
        }
        $plan = new SubscriptionPlan(); // For the form partial
        $moduleFeatures = [];
        $availableGateways = [];
        $targetRoles = [null => 'General (All Users)'] + Role::orderBy('name')->pluck('name', 'name')->all();

        if (setting('custom_payment_gateway_enabled', false)) { // Assuming a setting controls if custom gateway is an option
            $availableGateways['custom'] = setting('custom_payment_gateway_name', 'Custom Payment Instructions');
        }

        if (function_exists('setting') && setting('wallet_system_enabled', '0') == '1') {
            $availableGateways['wallet'] = __('Pay with Wallet Balance');
        }

        foreach (Module::allEnabled() as $module) {
            $features = $module->get('features'); // Assuming 'features' is an array in module.json
            if (is_array($features) && !empty($features)) {
                $moduleFeatures[$module->getName()] = $features;
            }

            // Check if the module is a payment gateway
            if ($module->get('type') === 'payment_gateway' || $module->get('category') === 'payment_gateway') {
                $moduleNameLower = strtolower($module->getName());
                $enabledSettingKey = null;

                // Determine the correct setting key for "enabled" status
                if ($moduleNameLower === 'paypalgateway') $enabledSettingKey = 'paypal_enabled';
                elseif ($moduleNameLower === 'paystackgateway') $enabledSettingKey = 'paystack_enabled';
                elseif ($moduleNameLower === 'stripegateway') $enabledSettingKey = 'stripe_enabled'; // Based on Stripe's config blade and processSubscription
                elseif ($moduleNameLower === 'authorizenetgateway') $enabledSettingKey = 'authorizenetgateway_enabled';
                else $enabledSettingKey = $moduleNameLower . '_enabled'; // Default fallback

                if (function_exists('setting') && setting($enabledSettingKey, false)) {
                    $gatewayKey = $module->get('gateway_key', Str::snake($module->getName()));
                    $availableGateways[$gatewayKey] = $module->get('display_name', $module->getName());
                }
            }
        }
        ksort($availableGateways); // Sort gateways by key for consistent order
        return view('admin.subscription_plans.create', compact('plan', 'moduleFeatures', 'availableGateways', 'targetRoles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (function_exists('setting') && setting('subscriptions_enabled', '0') != '1') {
            return redirect()->route('admin.dashboard')->with('info', 'Subscription feature is currently disabled.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'interval' => ['required', 'string', Rule::in(['day', 'week', 'month', 'year'])],
            'interval_count' => 'required|integer|min:1',
            'trial_period_days' => 'nullable|integer|min:0',
            'selected_features' => 'nullable|array',
            'selected_features.*' => 'string', // Keys of selected features
            'feature_limits' => 'nullable|array',
            'feature_limits.*' => 'nullable|integer|min:0', // Each limit, 0 means limit is set to 0
            'active' => 'boolean',
            'resource_limits' => 'nullable|array', // For global limits
            'resource_limits.*' => 'nullable|integer|min:0',
            'available_gateways_input' => 'nullable|array',
            'available_gateways_input.*' => 'string', // Selected gateway keys
            'target_role' => 'nullable|string|max:100', // Validate target_role
            'credits_awarded_on_purchase' => 'nullable|integer|min:0',
            'credits_awarded_on_renewal' => 'nullable|integer|min:0',
            'paypal_plan_id' => 'nullable|string|max:255',
            'stripe_product_id' => 'nullable|string|max:255',
            'stripe_price_id' => 'nullable|string|max:255',
            'paystack_plan_code' => 'nullable|string|max:255',
            'custom_gateway_plan_id' => 'nullable|string|max:255',
        ]);

        $validated['slug'] = Str::slug($validated['name'] . '-' . $validated['interval'] . ($validated['interval_count'] > 1 ? '-' . $validated['interval_count'] : ''));
        $originalSlug = $validated['slug'];
        $count = 1;
        while (SubscriptionPlan::where('slug', $validated['slug'])->exists()) {
            $validated['slug'] = $originalSlug . '-' . $count++;
        }

        $selectedFeatureKeys = $request->input('selected_features', []);
        $featureLimitsInput = $request->input('feature_limits', []);
        $featuresData = [];
        if (is_array($selectedFeatureKeys)) {
            foreach ($selectedFeatureKeys as $featureKey) {
                $featureItem = ['key' => $featureKey];
                if (isset($featureLimitsInput[$featureKey]) && $featureLimitsInput[$featureKey] !== '' && !is_null($featureLimitsInput[$featureKey])) {
                    $featureItem['limit'] = (int)$featureLimitsInput[$featureKey];
                }
                $featuresData[] = $featureItem;
            }
        }
        $uniqueFeaturesData = [];
        $seenKeys = [];
        foreach ($featuresData as $item) {
            if (!in_array($item['key'], $seenKeys)) {
                $uniqueFeaturesData[] = $item;
                $seenKeys[] = $item['key'];
            }
        }
        $validated['features'] = $uniqueFeaturesData;
        $validated['active'] = $request->has('active');

        $resourceLimits = $request->input('resource_limits', []);
        $validated['resource_limits'] = array_filter($resourceLimits, fn($value) => !is_null($value) && $value !== '');
        $validated['available_gateways'] = $request->input('available_gateways_input', []);
        $validated['credits_awarded_on_purchase'] = $request->input('credits_awarded_on_purchase', 0);
        $validated['credits_awarded_on_renewal'] = $request->input('credits_awarded_on_renewal', 0);

        DB::beginTransaction();
        try {
            SubscriptionPlan::create($validated);
            DB::commit();
            return redirect()->route('admin.subscription-plans.index')->with('success', 'Subscription plan created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating subscription plan: ' . $e->getMessage(), [
                'trace' => Str::limit($e->getTraceAsString(), 1000) // Limit trace length for logs
            ]);
            return redirect()->back()->withInput()->with('error', 'Failed to create subscription plan. Please try again.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(SubscriptionPlan $subscriptionPlan)
    {
        // It's common to redirect to edit for admin panels, or you can create a dedicated show view.
        // For now, let's keep it as redirecting to edit as per your existing structure.
        return view('admin.subscription_plans.edit', $this->getEditViewData($subscriptionPlan));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SubscriptionPlan $subscriptionPlan)
    {
        return view('admin.subscription_plans.edit', $this->getEditViewData($subscriptionPlan));
    }

    private function getEditViewData(SubscriptionPlan $subscriptionPlan)
    {
        if (function_exists('setting') && setting('subscriptions_enabled', '0') != '1') {
            // This will be caught by the view rendering if redirect happens before view() call
            // Consider throwing an exception or handling differently if called programmatically
            // For a direct view call, the view itself should handle this or the calling method.
        }
        $moduleFeatures = [];
        $availableGateways = [];
        $targetRoles = [null => 'General (All Users)'] + Role::orderBy('name')->pluck('name', 'name')->all();
        $currencies = \App\Helpers\CurrencyHelper::getCurrencies(); // Assuming you have this helper

        // Consolidate gateway and feature fetching logic (similar to create method)
        if (setting('custom_payment_gateway_enabled', false)) {
            $availableGateways['custom'] = setting('custom_payment_gateway_name', 'Custom Payment Instructions');
        }
        if (function_exists('setting') && setting('wallet_system_enabled', '0') == '1') {
            $availableGateways['wallet'] = __('Pay with Wallet Balance');
        }
        foreach (Module::allEnabled() as $module) {
            $features = $module->get('features');
            if (is_array($features) && !empty($features)) {
                $moduleFeatures[$module->getName()] = $features;
            }
            if ($module->get('type') === 'payment_gateway' || $module->get('category') === 'payment_gateway') {
                $moduleNameLower = strtolower($module->getName());
                $enabledSettingKey = match ($moduleNameLower) {
                    'paypalgateway' => 'paypal_enabled',
                    'paystackgateway' => 'paystack_enabled',
                    'stripegateway' => 'stripe_enabled',
                    'authorizenetgateway' => 'authorizenetgateway_enabled',
                    default => $moduleNameLower . '_enabled',
                };
                if (function_exists('setting') && setting($enabledSettingKey, false)) {
                    $gatewayKey = $module->get('gateway_key', Str::snake($module->getName()));
                    $availableGateways[$gatewayKey] = $module->get('display_name', $module->getName());
                }
            }
        }
        ksort($availableGateways);

        return compact('subscriptionPlan', 'moduleFeatures', 'availableGateways', 'targetRoles', 'currencies');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        if (function_exists('setting') && setting('subscriptions_enabled', '0') != '1') {
            return redirect()->route('admin.dashboard')->with('info', 'Subscription feature is currently disabled.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'interval' => ['required', 'string', Rule::in(['day', 'week', 'month', 'year'])],
            'interval_count' => 'required|integer|min:1',
            'trial_period_days' => 'nullable|integer|min:0',
            'selected_features' => 'nullable|array',
            'selected_features.*' => 'string', // Keys of selected features
            'feature_limits' => 'nullable|array',
            'feature_limits.*' => 'nullable|integer|min:0', // Each limit, 0 means limit is set to 0
            'active' => 'boolean',
            'resource_limits' => 'nullable|array', // For global limits
            'resource_limits.*' => 'nullable|integer|min:0',
            'available_gateways_input' => 'nullable|array',
            'available_gateways_input.*' => 'string',
            'target_role' => 'nullable|string|max:100',
            'credits_awarded_on_purchase' => 'nullable|integer|min:0',
            'credits_awarded_on_renewal' => 'nullable|integer|min:0',
            'paypal_plan_id' => 'nullable|string|max:255',
            'stripe_product_id' => 'nullable|string|max:255',
            'stripe_price_id' => 'nullable|string|max:255',
            'paystack_plan_code' => 'nullable|string|max:255',
            'custom_gateway_plan_id' => 'nullable|string|max:255',
        ]);

        $newSlug = Str::slug($validated['name'] . '-' . $validated['interval'] . ($validated['interval_count'] > 1 ? '-' . $validated['interval_count'] : ''));
        if ($newSlug !== $subscriptionPlan->slug) {
            $originalSlug = $newSlug;
            $count = 1;
            while (SubscriptionPlan::where('slug', $newSlug)->where('id', '!=', $subscriptionPlan->id)->exists()) {
                $newSlug = $originalSlug . '-' . $count++;
            }
            $validated['slug'] = $newSlug;
        }

        $selectedFeatureKeys = $request->input('selected_features', []);
        $featureLimitsInput = $request->input('feature_limits', []);
        $featuresData = [];
        if (is_array($selectedFeatureKeys)) {
            foreach ($selectedFeatureKeys as $featureKey) {
                $featureItem = ['key' => $featureKey];
                if (isset($featureLimitsInput[$featureKey]) && $featureLimitsInput[$featureKey] !== '' && !is_null($featureLimitsInput[$featureKey])) {
                    $featureItem['limit'] = (int)$featureLimitsInput[$featureKey];
                }
                $featuresData[] = $featureItem;
            }
        }
        $uniqueFeaturesData = [];
        $seenKeys = [];
        foreach ($featuresData as $item) {
            if (!in_array($item['key'], $seenKeys)) {
                $uniqueFeaturesData[] = $item;
                $seenKeys[] = $item['key'];
            }
        }
        $validated['features'] = $uniqueFeaturesData;

        $validated['active'] = $request->has('active');
        $resourceLimits = $request->input('resource_limits', []);
        $validated['resource_limits'] = array_filter($resourceLimits, fn($value) => !is_null($value) && $value !== '');
        $validated['available_gateways'] = $request->input('available_gateways_input', []);
        $validated['credits_awarded_on_purchase'] = $request->input('credits_awarded_on_purchase', 0);
        $validated['credits_awarded_on_renewal'] = $request->input('credits_awarded_on_renewal', 0);

        DB::beginTransaction();
        try {
            $subscriptionPlan->update($validated);
            DB::commit();
            return redirect()->route('admin.subscription-plans.index')->with('success', 'Subscription plan updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating subscription plan: ' . $e->getMessage(), [
                'plan_id' => $subscriptionPlan->id,
                'trace' => Str::limit($e->getTraceAsString(), 1000)
            ]);
            return redirect()->back()->withInput()->with('error', 'Failed to update subscription plan. Please try again.');
        }

        
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SubscriptionPlan $subscriptionPlan)
    {
        $subscriptionPlan->delete();
        return redirect()->route('admin.subscription-plans.index')->with('success', 'Subscription plan deleted successfully.');
    }
}