<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SettingGroupPermission;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Nwidart\Modules\Facades\Module; // Added for module interaction

class SettingController extends Controller
{
    private function getGroupDescriptions()
    {
        return [
            'General' => 'Manage core application settings including site identity, contact, mail, user registration, security, social logins, integrations, subscriptions, wallet, and credits system configurations.',
            'Payment Gateways' => 'Configure settings for various payment gateway integrations used across the application for subscriptions, wallet deposits, and other transactions.',
            'Webpilot AI' => 'Manage settings related to AI model providers, AI image generation, stock photo libraries, AI-assisted deployment, and other Webpilot AI features.',
            // You might keep old descriptions for a while if modules directly reference them,
            // or if you have a mechanism for modules to register their own group descriptions.
        ];
    }

    public function index(Request $request)
    {
        $allSettingsFromDb = Setting::orderBy('group')->orderBy('sub_group')->orderBy('name')->get(); // Order by group, sub_group, then name
        $currentUser = Auth::user();

        $coreSettingGroupKeys = [ // Define core groups explicitly by their key
            'General',
            'Payment Gateways',
            'Webpilot AI',
        ];

        $applicationSettings = collect(); // For core app settings: [GroupName => [SubGroupName => CollectionOfSettings]]
        $moduleSettings = collect();      // For module settings: [ModuleDisplayName => [GroupNameInModule => [SubGroupNameInModule => CollectionOfSettings]]]
        
        $activeMainTab = $request->query('tab', 'application'); // Default to application settings main tab
        $activeSubTab = $request->query('sub_tab'); // Will be determined based on activeMainTab
        foreach ($allSettingsFromDb->groupBy('group') as $originalGroupName => $settingsInGroup) {
            $canViewGroup = false;
            if ($currentUser->isSuperAdmin()) {
                $canViewGroup = true;
            } elseif ($currentUser->hasRole('admin')) {
                $canViewGroup = SettingGroupPermission::where('role_name', 'admin')
                                    ->where('setting_group_name', $originalGroupName)
                                    ->where('can_view', true)->exists();
            } else { // For other roles that might have direct permissions via RoleSettingGroupPermission
                // Assuming a user has one primary role for simplicity here. Adjust if multiple roles are common.
                $userRole = $currentUser->roles->first();
                if ($userRole) {
                    $canViewGroup = \App\Models\RoleSettingGroupPermission::where('role_id', $userRole->id)
                                        ->where('setting_group_name', $originalGroupName)
                                        ->where('can_view', true)->exists();
                }
            }

            if (!$canViewGroup) {
                continue; // Skip this group if user cannot view it
            }

            // Further group settings within this main group by their sub_group
            $settingsGroupedBySubGroup = $settingsInGroup->groupBy(function ($item) {
                return $item->sub_group ?? 'General Details'; // Default sub-group name if null or empty
            })->sortKeys(); // Sort sub-groups alphabetically within the main group

            // Categorize the group
            if (in_array($originalGroupName, $coreSettingGroupKeys)) {
                $applicationSettings[$originalGroupName] = $settingsGroupedBySubGroup;
            } elseif (str_contains($originalGroupName, '::')) {
                list($moduleKey, $groupNameInModule) = explode('::', $originalGroupName, 2);
                $moduleInstance = Module::find($moduleKey);

                // Only process module settings if the module exists and is enabled
                if ($moduleInstance && $moduleInstance->isEnabled()) {
                    $moduleDisplayName = $moduleInstance->get('display_name', $moduleKey);

                    if (!isset($moduleSettings[$moduleDisplayName])) {
                        $moduleSettings[$moduleDisplayName] = collect();
                    }
                    $moduleSettings[$moduleDisplayName][$groupNameInModule] = $settingsGroupedBySubGroup;
                } else {
                    // Module is not found or not enabled. These are orphaned settings.
                    // For now, we will not display them.
                    // Future enhancement: Super admin could have an "Orphaned Settings" view for cleanup.
                    Log::info("Skipping settings group '{$originalGroupName}' because module '{$moduleKey}' is not found or disabled.");
                }
            } else {
                // Fallback for groups not in core and not using '::' convention
                // Treat them as application settings for now, or create a "Miscellaneous" tab
                $applicationSettings[$originalGroupName] = $settingsGroupedBySubGroup;
            }
        }

        // Sort application settings by a predefined order if necessary, then by name
        $applicationSettings = $applicationSettings->sortBy(fn ($s, $gn) => array_search($gn, $coreSettingGroupKeys) === false ? 999 : array_search($gn, $coreSettingGroupKeys));
        $moduleSettings = $moduleSettings->sortKeys(); // Sort modules by display name

        // Prepare data for Super Admin to manage 'admin' role's setting group permissions
        $adminSettingPermissions = []; // For super_admin to manage 'admin' role permissions
        if ($currentUser->isSuperAdmin()) {
            // Get all group names present in the database for admin role permission management
            $allGroupNamesFromDb = Setting::distinct()->orderBy('group')->pluck('group');

            foreach ($allGroupNamesFromDb as $dbGroupName) {
                // Check if it's one of the new main application groups
                $isActualAppMainGroup = in_array($dbGroupName, ['General', 'Payment Gateways', 'Webpilot AI']);

                $isPotentiallyActiveModuleGroup = false;
                if (str_contains($dbGroupName, '::')) {
                    list($moduleKey) = explode('::', $dbGroupName, 2);
                    $moduleInstance = Module::find($moduleKey);
                    if ($moduleInstance && $moduleInstance->isEnabled()) {
                        $isPotentiallyActiveModuleGroup = true;
                    }
                }
                // Only add permissions for actual app main groups or active module groups
                if ($isActualAppMainGroup || $isPotentiallyActiveModuleGroup) {
                    $perm = SettingGroupPermission::firstOrNew(['role_name' => 'admin', 'setting_group_name' => $dbGroupName]);
                    $adminSettingPermissions[$dbGroupName] = [
                        'can_view' => (bool) $perm->can_view,
                        'can_edit' => (bool) $perm->can_edit,
                    ];
                }
            }
            ksort($adminSettingPermissions); // Sort by group name for display
        }

        // Main Navigation items
        $mainNavItems = collect(['application' => 'Application Settings']);
        foreach ($moduleSettings->keys() as $moduleDispName) {
            $mainNavItems[Str::slug($moduleDispName)] = $moduleDispName;
        }
        // Ensure activeMainTab is valid, fallback to first if not
        if (!$mainNavItems->has($activeMainTab) && $mainNavItems->isNotEmpty()) {
            $activeMainTab = $mainNavItems->keys()->first();
        }

        // Sub-navigation items for the "Application Settings" tab
        $applicationSettingNavItems = collect();
        if ($activeMainTab === 'application') {
            if ($currentUser->isSuperAdmin() && !empty($adminSettingPermissions)) { // Only add if there are permissions to show
                $applicationSettingNavItems['permissions'] = 'Admin Role Permissions';
            }
            foreach ($applicationSettings->keys() as $groupName) {
                $applicationSettingNavItems[Str::slug($groupName)] = $groupName;
            }
            
            // Set default activeSubTab if not provided or invalid for the application tab
            if ((!$activeSubTab || !$applicationSettingNavItems->has($activeSubTab)) && $applicationSettingNavItems->isNotEmpty()) {
                $activeSubTab = $applicationSettingNavItems->keys()->first();
            } elseif ($applicationSettingNavItems->isEmpty()) {
                $activeSubTab = null; // No sub-tabs available
            }
        } else {
            $activeSubTab = null; // No sub-tabs for module settings main tabs
        }

        return view('admin.settings.index', [
            'applicationSettings' => $applicationSettings,
            'moduleSettings' => $moduleSettings,
            'adminSettingPermissions' => $adminSettingPermissions,
            'mainNavItems' => $mainNavItems, // Changed from navItems
            'applicationSettingNavItems' => $applicationSettingNavItems, // New
            'activeMainTab' => $activeMainTab, // Changed from activeTab
            'activeSubTab' => $activeSubTab,   // New
            'groupDescriptions' => $this->getGroupDescriptions(),
        ]);
    }

    public function update(Request $request)
    {
        $currentUser = Auth::user();

        if ($currentUser->isSuperAdmin() && $request->has('setting_group_permissions')) {
            $permissionsInput = $request->input('setting_group_permissions', []); // These are original group names
            $allDbGroupNames = Setting::distinct()->pluck('group');

            foreach ($allDbGroupNames as $dbGroupName) {
                SettingGroupPermission::updateOrCreate(
                    ['role_name' => 'admin', 'setting_group_name' => $dbGroupName],
                    [
                        'can_view' => isset($permissionsInput[$dbGroupName]['can_view']),
                        'can_edit' => isset($permissionsInput[$dbGroupName]['can_edit']),
                    ]
                );
            }
            if (!$request->has('settings_keys')) { // If only permissions were updated
                if (function_exists('clear_settings_cache')) { clear_settings_cache(); }
                return redirect()->route('admin.settings.index')->with('success', 'Admin role setting permissions updated successfully!');
            }
        }

        $validated = $request->validate([
            'settings_keys' => 'sometimes|required|array', // 'sometimes' if only permissions are updated
            'settings_keys.*' => 'required_with:settings_keys|string|exists:settings,key',
            'settings_values' => 'nullable|array',
            'settings_values.*' => 'nullable|string',
            'settings_files' => 'nullable|array',
            'settings_files.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,ico,webp|max:4096', // Increased max size
        ]);

        if (isset($validated['settings_keys'])) {
            foreach ($validated['settings_keys'] as $settingKey) {
                $setting = Setting::where('key', $settingKey)->firstOrFail();

                if (!$currentUser->isSuperAdmin() && $currentUser->hasRole('admin')) {
                    $canEditGroup = SettingGroupPermission::where('role_name', 'admin')
                                        ->where('setting_group_name', $setting->group)
                                        ->where('can_edit', true)->exists();
                    if (!$canEditGroup && ($request->has('settings_values.' . $settingKey) || $request->hasFile('settings_files.' . $settingKey))) {
                        // If only trying to update a setting in a non-editable group,
                        // you might throw an error or just silently skip.
                        return redirect()->route('admin.settings.index')->with('error', "You do not have permission to edit settings in the '{$setting->group}' group.");
                    }
                }

                $newValue = $request->input('settings_values.' . $settingKey);

                if ($request->hasFile('settings_files.' . $settingKey)) {
                    $file = $request->file('settings_files.' . $settingKey);
                    if ($file->isValid()) {
                        if ($setting->value && Storage::disk('public')->exists($setting->value)) {
                            Storage::disk('public')->delete($setting->value);
                        }
                        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                        $filename = Str::slug($originalName) . '_' . time() . '.' . $file->getClientOriginalExtension();
                        $path = $file->storeAs('settings_uploads', $filename, 'public');
                        $setting->update(['value' => $path]);
                    }
                } elseif ($setting->type === 'password') {
                    if (!empty($newValue)) {
                        $setting->update(['value' => Crypt::encryptString($newValue)]);
                    }
                } elseif ($setting->type === 'checkbox') {
                    $setting->update(['value' => $request->has('settings_values.' . $settingKey) && $request->input('settings_values.' . $settingKey) == '1' ? '1' : '0']);
                } else {
                    if ($request->has('settings_values.' . $settingKey) || !$request->hasFile('settings_files.' . $settingKey)) {
                         $setting->update(['value' => $newValue]);
                    }
                }
            }
        }

        if (function_exists('clear_settings_cache')) {
            clear_settings_cache();
        }

        return redirect()->route('admin.settings.index')->with('success', 'Settings updated successfully!');
    }

    public function testMail(Request $request)
    {
        $recipientEmail = Auth::user()->email;
        $siteName = setting('site_name', config('app.name'));

        try {
            Mail::raw("This is a test email from your application '{$siteName}'.\n\nIf you received this, your mail settings are configured correctly.", function ($message) use ($recipientEmail, $siteName) {
                $message->to($recipientEmail)
                        ->subject('Test Email Configuration - ' . $siteName);
            });
            Log::info('Test email sent successfully to: ' . $recipientEmail);
            return response()->json(['success' => true, 'message' => 'Test email sent successfully to ' . $recipientEmail . '!']);
        } catch (\Exception $e) {
            Log::error('Failed to send test email to: ' . $recipientEmail . '. Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Failed to send test email: ' . $e->getMessage()], 500);
        }
    }
}
